package controllers

import play.api.Routes
import play.api.mvc._
import play.api.Play
import play.api.Logger
import play.api.Play.current
import play.api.data.Form
import play.api.data.Forms._
import play.api.i18n.{Messages, Lang}
import play.api.libs.json.{JsObject, Json}
import play.api.libs.concurrent.Execution.Implicits._

import scala.concurrent.Future
import scala.Some

import util.Failure

import java.io.File

import models.{User, ContentModel}
import helpers.{UUID, JsMessages}

object Tagger extends Controller with Secured {

  val messages = JsMessages.default

  val allJsMessages = Action {
    Ok(messages.all(Some("window.Messages")))
  }

  def javascriptRoutes = Action { implicit request =>
    import routes.javascript._
    Ok(
      Routes.javascriptRouter("jsRoutes")(
        routes.javascript.Application.sendImage,
        routes.javascript.Tagger.delete,
        routes.javascript.Tagger.upload,
        controllers.api.routes.javascript.Audios.addAudio,
        controllers.api.routes.javascript.Audios.editAudio,
        controllers.api.routes.javascript.Contents.findByLanguage,
        controllers.api.routes.javascript.Contents.findById,
        controllers.api.routes.javascript.Contents.findUntagged,
        controllers.api.routes.javascript.Contents.findByTags,
        controllers.api.routes.javascript.Contents.isSlugUnique,
        controllers.api.routes.javascript.Featureds.findByLanguage,
        controllers.api.routes.javascript.Featureds.editFeatured,
        controllers.api.routes.javascript.Featureds.addFeatured,
        controllers.api.routes.javascript.FooterItems.findByLanguage,
        controllers.api.routes.javascript.FooterItems.editFooterItem,
        controllers.api.routes.javascript.FooterItems.addFooterItem,
        controllers.api.routes.javascript.HeaderItems.findByLanguage,
        controllers.api.routes.javascript.HeaderItems.setPosition,
        controllers.api.routes.javascript.HeaderItems.editHeaderItem,
        controllers.api.routes.javascript.HeaderItems.addHeaderItem,
        controllers.api.routes.javascript.HomeBlocks.findByLanguage,
        controllers.api.routes.javascript.HomeBlocks.save,
        controllers.api.routes.javascript.Links.addLink,
        controllers.api.routes.javascript.Links.editLink,
        controllers.api.routes.javascript.MenuItems.findAllParents,
        controllers.api.routes.javascript.MenuItems.findByParentId,
        controllers.api.routes.javascript.MenuItems.setPosition,
        controllers.api.routes.javascript.MenuItems.create,
        controllers.api.routes.javascript.MenuItems.save,
        controllers.api.routes.javascript.MenuItems.deleteMenuItem,
        controllers.api.routes.javascript.SelectFeatureds.findByLanguage,
        controllers.api.routes.javascript.SelectFeatureds.editSelectFeatured,
        controllers.api.routes.javascript.SelectFeatureds.addSelectFeatured,
        controllers.api.routes.javascript.SiteWords.findByLanguage,
        controllers.api.routes.javascript.SiteWords.setPosition,
        controllers.api.routes.javascript.SiteWords.editSiteWord,
        controllers.api.routes.javascript.SiteWords.addSiteWord,
        controllers.api.routes.javascript.Sites.addSite,
        controllers.api.routes.javascript.Sites.editSite,
        controllers.api.routes.javascript.Slides.findByLanguage,
        controllers.api.routes.javascript.Slides.addSlide,
        controllers.api.routes.javascript.Slides.editSlide,
        controllers.api.routes.javascript.TagLines.findByLanguage,
        controllers.api.routes.javascript.TagLines.setPosition,
        controllers.api.routes.javascript.TagLines.addTagLine,
        controllers.api.routes.javascript.TagLines.editTagLine,
        controllers.api.routes.javascript.Tags.findByLanguage,
        controllers.api.routes.javascript.Tags.findByTagged,
        controllers.api.routes.javascript.Tags.findByManyTagged,
        controllers.api.routes.javascript.Tags.findByTags,
        controllers.api.routes.javascript.Tags.findById,
        controllers.api.routes.javascript.Tags.editTag,
        controllers.api.routes.javascript.Tags.deleteTag,
        controllers.api.routes.javascript.Texts.addText,
        controllers.api.routes.javascript.Texts.editText,
        controllers.api.routes.javascript.Videos.addVideo,
        controllers.api.routes.javascript.Videos.editVideo
      )
    ).as("text/javascript")
  }

  /**
   * Form for user authorization.
   */
  def authForm = Form(
    tuple(
      "email" -> text,
      "password" -> text
    )
  )

  /**
   * Show the administrative panel.
   */
  def admin = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    Future(Ok(views.html.admin.index("Admin")))
  }

  /**
   * Login form.
   *
   * @return
   */
  def login = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    Future(Ok(views.html.admin.login("Admin Login")))
  }

  /**
   * Logs out a user and destroys their session.
   */
  def logout = Action { implicit request =>
    Redirect("/").withNewSession.flashing(
      "success" -> Messages("logout.success")
    )
  }

  /**
   * Change language of site.
   */
  def language(lang: String) = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    val referrer = request.headers.get(REFERER).getOrElse("/")
    Future(Redirect(referrer).withLang(Lang(lang)).asInstanceOf[SimpleResult])
  }

  /**
   * Login form submission.
   *
   * Verifies the login form and, if valid, it attempts to authenticate
   * the user.
   *
   * @return
   */
  def authenticate = Action.async { implicit request =>
    authForm.bindFromRequest.fold(
      formWithErrors =>
        Future.successful(Redirect(routes.Tagger.login)
          .withNewSession
          .flashing("error" -> Messages("login.badRequest"))),

      authTuple => {
        val email = authTuple._1
        val password = authTuple._2

        User.authenticate(email, password).map {
          case Some(user) => Redirect(routes.Tagger.admin)
                              .withSession("user_id" -> user.id.string)
                              .flashing("success" -> Messages("login.success"))
          case None => Redirect(routes.Tagger.login).withNewSession.flashing("error" -> Messages("login.notFound"))
        }
      }
    )
  }

  /**
   * Uploads a file on the server under the "uploads" directory.
   */
  def upload = IsAuthenticatedAsync(parse.multipartFormData) { implicit maybeUser => implicit request =>
    request.body.file("upload").map { picture =>
      val fileName = picture.filename.split("\\.").head
      val fileType = picture.filename.split("\\.").last

      val uploadFile = fileName + UUID.random.string + "." + fileType

      picture.ref.moveTo(new File(Play.current.configuration.getString("uploads.outFile").get + uploadFile))
      Future(Ok(Json.toJson(uploadFile)))
    }.getOrElse {
      Future(Ok(Json.toJson("FAILED")))
    }
  }

  def delete(id: String) = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    ContentModel.delete(UUID(id)).map { result =>
      result match {
        case true => Redirect(routes.Tagger.admin).flashing("success" -> (Messages("index.delete")))
        case false => Redirect(routes.Tagger.admin).flashing("error" -> (Messages("index.badRequest")))
      }
    }
  }

}

/**
 * Provide security features
 */
trait Secured {

  /**
   * Retrieve the connected user ID.
   */
  private def userId(request: RequestHeader) = request.session.get("user_id")


  /**
   * Redirect to login if the user in not authorized.
   */
  private def onUnauthorized(request: RequestHeader) = Results.Redirect(routes.Tagger.login)

  def CheckIfAuthenticated[A](userinfo: RequestHeader => Option[A])(action: Option[A] => EssentialAction): EssentialAction = {

    EssentialAction { request =>
      userinfo(request).map { user =>
        action(Some(user))(request)
      }.getOrElse {
        action(None)(request)
      }
    }
  }

  /**
   * Action for authenticated users.
   */
  def IsAuthenticated(f: => String => Request[AnyContent] => Result) = Security.Authenticated(userId, onUnauthorized) { user =>
    Action(request => f(user)(request))
  }

  /**
   * Action for possibly authenticated users.
   */
  def MaybeAuthenticated(f: => Option[String] => Request[AnyContent] => Result) = CheckIfAuthenticated(userId) { maybeUserId =>
    Action(request => f(maybeUserId)(request))
  }

  /**
   * Asynchronous version of IsAuthenticated that loads up the user entity.
   */
  def IsAuthenticatedAsync[A](bodyParser: BodyParser[A])(f: => User => Request[A] => Future[SimpleResult]) = Security.Authenticated(userId, onUnauthorized) { userId =>
    Action.async(bodyParser) { request =>
        User.find(userId).flatMap {
          case Some(user) => f(user)(request)
          case None => Future(Results.Redirect(routes.Tagger.login))
        }
    }
  }

  /**
   * Asynchronous version of MaybeAuthenticated that loads up the user entity.
   */
  def MaybeAuthenticatedAsync(f: => Option[User] => Request[AnyContent] => Future[SimpleResult]) = CheckIfAuthenticated(userId) { maybeUserId =>
    Action.async { request =>
      maybeUserId match {
        // User ID was given
        case Some(userId) => {
          User.find(userId).flatMap {
            case Some(user) => f(Some(user))(request)
            case None => f(None)(request)
          }
        }
        // Anonymous user
        case None => f(None)(request)
      }
    }
  }

}
