package controllers.api

import play.api.mvc.Controller
import play.api.cache.Cache
import play.api.data.Form
import play.api.data.Forms._
import play.api.i18n.Messages
import play.api.libs.concurrent.Execution.Implicits._

import scala.concurrent.Future
import scala.Some

import models.{Audio, AudioFormData, Tag}
import helpers.{UUID, Util}

object Audios extends Controller with controllers.Secured {

  def audioForm = Form(
    mapping(
      "english-title" -> text,
      "french-title" -> text,
      "english-slug" -> text,
      "french-slug" -> text,
      "english-description" -> text,
      "french-description" -> text,
      "english_audio_url" -> text,
      "french_audio_url" -> text,
      "english-primary-tag" -> text,
      "french-primary-tag" -> text,
      "english-tags" -> text,
      "french-tags" -> text,
      "starred" -> number
    )(AudioFormData.apply)(AudioFormData.unapply)
  )

  def addAudio = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    audioForm.bindFromRequest.fold(
      formWithErrors => Future.successful(Ok(Messages("admin.badRequest"))),

      audioFormData => {
        val englishId = UUID.random
        val frenchId = UUID.random
        val englishPrimaryTag = audioFormData.english_tags.split(",").distinct(0).trim
        val frenchPrimaryTag = audioFormData.french_tags.split(",").distinct(0).trim
        val englishSlug = {
          if (audioFormData.english_slug == "") {
            Util.slugify(englishPrimaryTag, audioFormData.english_title)
          }
          else {
            Util.slugify(englishPrimaryTag, audioFormData.english_slug)
          }
        }
        val frenchSlug = {
          if (audioFormData.french_slug == "") {
            Util.slugify(frenchPrimaryTag, audioFormData.french_title)
          }
          else {
            Util.slugify(frenchPrimaryTag, audioFormData.french_slug)
          }
        }

        val futureEnglishAudio = Audio.create(new Audio(  id = englishId,
                                                          translated_id = frenchId,
                                                          title = audioFormData.english_title,
                                                          slug = englishSlug,
                                                          description = audioFormData.english_description,
                                                          primary_tag = englishPrimaryTag,
                                                          starred = audioFormData.starred,
                                                          audio_url = audioFormData.english_audio_url), "english", user)

        val futureFrenchAudio = Audio.create(new Audio( id = frenchId,
                                                        translated_id = englishId,
                                                        title = audioFormData.french_title,
                                                        slug = frenchSlug,
                                                        description = audioFormData.french_description,
                                                        primary_tag = frenchPrimaryTag,
                                                        starred = audioFormData.starred,
                                                        audio_url = audioFormData.french_audio_url), "french", user)

        for {
          maybeEnglishAudio <- futureEnglishAudio
          maybeFrenchAudio <- futureFrenchAudio
        }
        yield {
          (maybeEnglishAudio, maybeFrenchAudio) match {
            case (Some(english), Some(french)) => {
              Tag.tagContent(audioFormData.english_tags, english.id.bytes, "english")
              Tag.tagContent(audioFormData.french_tags, french.id.bytes, "french")
            }
            case _ => {}
          }
          Redirect(controllers.routes.Tagger.admin).flashing("success" -> Messages("admin.audio-added"))
        }
      }
    )
  }

  def editAudio(english_id: String, french_id: String) = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    audioForm.bindFromRequest.fold(
      formWithErrors => Future.successful(Redirect(controllers.routes.Tagger.admin).flashing("error" -> Messages("admin.badRequest"))),

      audioFormData => {
        val englishSlug = {
          if (audioFormData.english_slug == "") {
            Util.slugify(audioFormData.english_primary_tag, audioFormData.english_title)
          }
          else {
            audioFormData.english_slug
          }
        }
        val frenchSlug = {
          if (audioFormData.french_slug == "") {
            Util.slugify(audioFormData.french_primary_tag, audioFormData.french_title)
          }
          else {
            audioFormData.french_slug
          }
        }

        val futureEnglishAudio = Audio.update(new Audio( id = UUID(english_id),
                                                      translated_id = UUID.random,
                                                      title = audioFormData.english_title,
                                                      slug = englishSlug,
                                                      description = audioFormData.english_description,
                                                      primary_tag = audioFormData.english_primary_tag,
                                                      starred = audioFormData.starred,
                                                      audio_url = audioFormData.english_audio_url), user)

        val futureFrenchAudio = Audio.update(new Audio(id = UUID(french_id),
                                                    translated_id = UUID.random,
                                                    title = audioFormData.french_title,
                                                    slug = frenchSlug,
                                                    description = audioFormData.french_description,
                                                    primary_tag = audioFormData.french_primary_tag,
                                                    starred = audioFormData.starred,
                                                    audio_url = audioFormData.french_audio_url), user)

        for {
          maybeEnglishAudio <- futureEnglishAudio
          maybeFrenchAudio <- futureFrenchAudio
        }
        yield {
          (maybeEnglishAudio, maybeFrenchAudio) match {
            case (Some(english), Some(french)) => {
              Tag.tagContent(audioFormData.english_tags, UUID(english_id).bytes, "english")
              Tag.tagContent(audioFormData.french_tags, UUID(french_id).bytes, "french")
            }
            case _ => {}
          }
          Redirect("/admin#/edit/tagged/content").flashing("success" -> (Messages("admin.audio-edited")))
        }
      }
    )
  }

}
