package controllers.api

import play.api.mvc.Controller
import play.api.cache.Cache
import play.api.data.Form
import play.api.data.Forms._
import play.api.i18n.Messages
import play.api.libs.json.{JsObject, Json}
import play.api.libs.concurrent.Execution.Implicits._

import scala.concurrent.duration._
import scala.concurrent.{Future, Await}
import scala.Some

import util.Failure

import java.io.File

import models._
import helpers.UUID

object Contents extends Controller with controllers.Secured {

  def findByLanguage(language: String) = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    ContentModel.findByLanguage(language).map { contents =>
      Ok(Json.toJson(contents.map { content =>
        content.content_type match {
          case "text" => Json.toJson(content.asInstanceOf[Text])

          case "video" => Json.toJson(content.asInstanceOf[Video])

          case "link" => Json.toJson(content.asInstanceOf[Link])

          case "audio" => Json.toJson(content.asInstanceOf[Audio])

          case _ => {
            Json.obj("id" -> content.id.string,
                     "translated_id" -> content.translated_id.string,
                     "title"  -> content.title,
                     "description" -> content.description,
                     "content_type" -> content.content_type,
                     "created_at" -> content.createdAt)
          }
        }
      }))
    }
  }

  def findById(id: String) = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    ContentModel.find(UUID(id)).flatMap { content =>
      ContentModel.findLanguage(UUID(id)).map { language =>
        Ok(Json.toJson(content.map { content =>

          content.content_type match {
            case "text" => {
              var text = content.asInstanceOf[Text]

              Json.obj("id" -> text.id.string,
                       "translated_id" -> content.translated_id.string,
                       "title"  -> text.title,
                       "slug" -> text.slug,
                       "description" -> text.description,
                       "content_type" -> text.content_type,
                       "primary_tag" -> text.primary_tag,
                       "starred" -> text.starred.asInstanceOf[Int],
                       "content" -> text.content,
                       "created_at" -> text.createdAt,
                       "language" -> language)
            }

            case "video" => {
              var video = content.asInstanceOf[Video]

              Json.obj("id" -> content.id.string,
                       "translated_id" -> content.translated_id.string,
                       "title"  -> video.title,
                       "slug" -> video.slug,
                       "description" -> video.description,
                       "long_description" -> video.long_description,
                       "content_type" -> video.content_type,
                       "primary_tag" -> video.primary_tag,
                       "starred" -> video.starred.asInstanceOf[Int],
                       "video_type" -> video.video_type,
                       "video_url" -> video.video_url,
                       "created_at" -> video.createdAt,
                       "language" -> language)
            }

            case "link" => {
              var link = content.asInstanceOf[Link]

              Json.obj("id" -> link.id.string,
                       "translated_id" -> content.translated_id.string,
                       "title"  -> link.title,
                       "slug" -> link.slug,
                       "description" -> link.description,
                       "content_type" -> link.content_type,
                       "primary_tag" -> link.primary_tag,
                       "starred" -> link.starred.asInstanceOf[Int],
                       "link_url" -> link.link_url,
                       "created_at" -> link.createdAt,
                       "language" -> language)
            }

            case "audio" => {
              var audio = content.asInstanceOf[Audio]

              Json.obj("id" -> audio.id.string,
                       "translated_id" -> content.translated_id.string,
                       "title" -> audio.title,
                       "slug" -> audio.slug,
                       "description" -> audio.description,
                       "content_type" -> audio.content_type,
                       "primary_tag" -> audio.primary_tag,
                       "starred" -> audio.starred.asInstanceOf[Int],
                       "audio_url" -> audio.audio_url,
                       "created_at" -> audio.createdAt,
                       "language" -> language)
            }

            case "tag_line" => {
              var tagLine = content.asInstanceOf[TagLine]

              Json.obj("id" -> tagLine.id.string,
                       "translated_id" -> tagLine.translated_id.string,
                       "title" -> tagLine.title,
                       "description" -> tagLine.description,
                       "tag_line_position" -> tagLine.position.asInstanceOf[Int],
                       "content_type" -> tagLine.content_type,
                       "created_at" -> tagLine.createdAt,
                       "language" -> language)
            }

            case "header_item" => {
              var headerItem = content.asInstanceOf[HeaderItem]

              Json.obj("id" -> headerItem.id.string,
                       "translated_id" -> headerItem.translated_id.string,
                       "title" -> headerItem.title,
                       "description" -> headerItem.description,
                       "header_item_position" -> headerItem.position.asInstanceOf[Int],
                       "header_content" -> headerItem.content,
                       "content_type" -> headerItem.content_type,
                       "created_at" -> headerItem.createdAt,
                       "language" -> language)
            }

            case "footer_item" => {
              val footerItem = content.asInstanceOf[FooterItem]

              Json.obj("id" -> footerItem.id.string,
                       "translated_id" -> footerItem.translated_id.string,
                       "title" -> footerItem.title,
                       "description" -> footerItem.description,
                       "footer_item_position" -> footerItem.position.asInstanceOf[Int],
                       "footer_content" -> footerItem.content,
                       "content_type" -> footerItem.content_type,
                       "created_at" -> footerItem.createdAt,
                       "language" -> language)
            }

            case "featured" => {
              var featured = content.asInstanceOf[Featured]

              Json.obj("id" -> featured.id.string,
                       "translated_id" -> featured.translated_id.string,
                       "title" -> featured.title,
                       "slug" -> featured.slug,
                       "description" -> featured.description,
                       "featured_position" -> featured.position.asInstanceOf[Int],
                       "icon_type" -> featured.icon_type,
                       "featured_content" -> featured.content,
                       "content_type" -> featured.content_type,
                       "primary_tag" -> featured.primary_tag,
                       "starred" -> featured.starred.asInstanceOf[Int],
                       "created_at" -> featured.createdAt,
                       "language" -> language)
            }

            case "select_featured" => {
              var selectFeatured = content.asInstanceOf[SelectFeatured]

              Json.obj("id" -> selectFeatured.id.string,
                       "translated_id" -> selectFeatured.translated_id.string,
                       "title" -> selectFeatured.title,
                       "description" -> selectFeatured.description,
                       "select_featured_position" -> selectFeatured.position.asInstanceOf[Int],
                       "select_featured_href_slug" -> selectFeatured.href_slug,
                       "select_featured_icon_type" -> selectFeatured.icon_type,
                       "content_type" -> selectFeatured.content_type,
                       "primary_tag" -> selectFeatured.primary_tag,
                       "starred" -> selectFeatured.starred.asInstanceOf[Int],
                       "created_at" -> selectFeatured.createdAt,
                       "language" -> language)
            }

            case "site_word" => {
              var siteWord = content.asInstanceOf[SiteWord]

              Json.obj("id" -> siteWord.id.string,
                       "translated_id" -> siteWord.translated_id.string,
                       "title" -> siteWord.title,
                       "description" -> siteWord.description,
                       "site_word_position" -> siteWord.position.asInstanceOf[Int],
                       "site_word_href_id" -> siteWord.href_id.string,
                       "content_type" -> siteWord.content_type,
                       "primary_tag" -> siteWord.primary_tag,
                       "created_at" -> siteWord.createdAt,
                       "language" -> language)
            }

            case "site" => {
              val site = content.asInstanceOf[Site]

              Json.obj("id" -> site.id.string,
                       "translated_id" -> site.translated_id.string,
                       "title" -> site.title,
                       "slug" -> site.slug,
                       "description" -> site.description,
                       "site_content" -> site.content,
                       "site_url" -> site.site_url,
                       "content_type" -> site.content_type,
                       "primary_tag" -> site.primary_tag,
                       "created_at" -> site.createdAt,
                       "language" -> language)
            }

            case "slide" => {
              val slide = content.asInstanceOf[Slide]

              Json.obj("id" -> slide.id.string,
                       "translated_id" -> slide.translated_id.string,
                       "title" -> slide.title,
                       "slug" -> slide.slug,
                       "description" -> slide.description,
                       "slide_position" -> slide.position.asInstanceOf[Int],
                       "slide_content" -> slide.content,
                       "slide_background" -> slide.slide_background,
                       "slide_url" -> slide.slide_url,
                       "content_type" -> slide.content_type,
                       "primary_tag" -> slide.primary_tag,
                       "created_at" -> slide.createdAt,
                       "language" -> language)
            }

            case _ => {
              Json.obj("id" -> content.id.string,
                       "translated_id" -> content.translated_id.string,
                       "title"  -> content.title,
                       "description" -> content.description,
                       "content_type" -> content.content_type,
                       "primary_tag" -> content.primary_tag,
                       "created_at" -> content.createdAt,
                       "language" -> language)
            }
          }
        }))
      }
    }
  }

  def findUntagged(language: String) = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    ContentModel.findUntagged(language).map { contents =>
      Ok(Json.toJson(contents.map { content =>
        content.content_type match {
          case "text" => Json.toJson(content.asInstanceOf[Text])

          case "video" => Json.toJson(content.asInstanceOf[Video])

          case "link" => Json.toJson(content.asInstanceOf[Link])

          case "audio" => Json.toJson(content.asInstanceOf[Audio])

          case "tag_line" => Json.toJson(content.asInstanceOf[TagLine])

          case _ => {
            Json.obj("id" -> content.id.string,
                     "translated_id" -> content.translated_id.string,
                     "title"  -> content.title,
                     "description" -> content.description,
                     "content_type" -> content.content_type,
                     "created_at" -> content.createdAt)
          }
        }
      }))
    }
  }

  def findByTags(tags: String, language: String) = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    ContentModel.findByTags(tags, language).map { content =>
      Ok(Json.toJson(content.map { content =>

        content.content_type match {
          case "text" => {
            var text = content.asInstanceOf[Text]

            Json.obj("id" -> text.id.string,
                     "translated_id" -> content.translated_id.string,
                     "title"  -> text.title,
                     "slug" -> text.slug,
                     "description" -> text.description,
                     "content_type" -> text.content_type,
                     "primary_tag" -> text.primary_tag,
                     "starred" -> text.starred.asInstanceOf[Int],
                     "content" -> text.content,
                     "created_at" -> text.createdAt)
          }

          case "video" => {
            var video = content.asInstanceOf[Video]

            Json.obj("id" -> content.id.string,
                     "translated_id" -> content.translated_id.string,
                     "title"  -> video.title,
                     "slug" -> video.slug,
                     "description" -> video.description,
                     "long_description" -> video.long_description,
                     "content_type" -> video.content_type,
                     "primary_tag" -> video.primary_tag,
                     "starred" -> video.starred.asInstanceOf[Int],
                     "video_type" -> video.video_type,
                     "video_url" -> video.video_url,
                     "created_at" -> video.createdAt)
          }

          case "link" => {
            var link = content.asInstanceOf[Link]

            Json.obj("id" -> link.id.string,
                     "translated_id" -> content.translated_id.string,
                     "title"  -> link.title,
                     "slug" -> link.slug,
                     "description" -> link.description,
                     "content_type" -> link.content_type,
                     "primary_tag" -> link.primary_tag,
                     "starred" -> link.starred.asInstanceOf[Int],
                     "link_url" -> link.link_url,
                     "created_at" -> link.createdAt)
          }

          case "audio" => {
            var audio = content.asInstanceOf[Audio]

            Json.obj("id" -> audio.id.string,
                     "translated_id" -> content.translated_id.string,
                     "title" -> audio.title,
                     "slug" -> audio.slug,
                     "description" -> audio.description,
                     "content_type" -> audio.content_type,
                     "primary_tag" -> audio.primary_tag,
                     "starred" -> audio.starred.asInstanceOf[Int],
                     "audio_url" -> audio.audio_url,
                     "created_at" -> audio.createdAt)
          }

          case _ => {
            Json.obj("id" -> content.id.string,
                     "translated_id" -> content.translated_id.string,
                     "title"  -> content.title,
                     "description" -> content.description,
                     "content_type" -> content.content_type,
                     "primary_tag" -> content.primary_tag,
                     "created_at" -> content.createdAt)
          }
        }
      }))
    }
  }

  def isSlugUnique(primaryTag: String, slug: String) = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    var subDir = {
      if (primaryTag != "none") {
        primaryTag + "/"
      }
      else {
        ""
      }
    }
    ContentModel.findBySlug(subDir + slug.trim().toLowerCase()).map {
      case Some(content) => Ok
      case None => NotFound
    }
  }
}
