package controllers.api

import play.api.mvc.Controller
import play.api.cache.Cache
import play.api.data.Form
import play.api.data.Forms._
import play.api.i18n.Messages
import play.api.libs.json.{JsObject, Json}
import play.api.libs.concurrent.Execution.Implicits._

import scala.concurrent.Future
import scala.Some

import models.{Featured, FeaturedFormData, Tag}
import helpers.{UUID, Util}

object Featureds extends Controller with controllers.Secured {

  def findByLanguage(language: String) = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    Featured.findByLanguage(language).map { contents =>
        Ok(Json.toJson(contents.map { featuredContent =>
          Json.obj("id" -> featuredContent.id.string,
                   "translated_id" -> featuredContent.translated_id.string,
                   "title" -> featuredContent.title,
                   "slug" -> featuredContent.slug,
                   "description" -> featuredContent.description,
                   "featured_position" -> featuredContent.position.asInstanceOf[Int],
                   "featured_content" -> featuredContent.content,
                   "icon_type" -> featuredContent.icon_type,
                   "content_type" -> featuredContent.content_type,
                   "primary_tag" -> featuredContent.primary_tag,
                   "starred" -> featuredContent.starred.asInstanceOf[Int],
                   "created_at" -> featuredContent.createdAt,
                   "language" -> language)
        }))
    }
  }

  def featuredForm = Form(
    mapping(
      "english-title" -> text,
      "french-title" -> text,
      "english-slug" -> text,
      "french-slug" -> text,
      "english-description" -> text,
      "french-description" -> text,
      "position" -> number,
      "icon_type" -> text,
      "english-content" -> text,
      "french-content" -> text,
      "english-primary-tag" -> text,
      "french-primary-tag" -> text,
      "starred" -> number
    )(FeaturedFormData.apply)(FeaturedFormData.unapply)
  )

  def addFeatured = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    featuredForm.bindFromRequest.fold(
      formWithErrors => Future.successful(Ok(Messages("admin.badRequest"))),

      featuredFormData => {
        val englishId = UUID.random
        val frenchId = UUID.random
        val englishPrimaryTag = featuredFormData.english_primary_tag
        val frenchPrimaryTag = featuredFormData.french_primary_tag
        val englishSlug = {
          if (featuredFormData.english_slug == "") {
            Util.slugify(englishPrimaryTag, featuredFormData.english_title)
          }
          else {
            Util.slugify(englishPrimaryTag, featuredFormData.english_slug)
          }
        }
        val frenchSlug = {
          if (featuredFormData.french_slug == "") {
            Util.slugify(frenchPrimaryTag, featuredFormData.french_title)
          }
          else {
            Util.slugify(frenchPrimaryTag, featuredFormData.french_slug)
          }
        }

        val futureEnglishFeatured = Featured.create(new Featured( id = englishId,
                                                                  translated_id = frenchId,
                                                                  title = featuredFormData.english_title,
                                                                  slug = englishSlug,
                                                                  description = featuredFormData.english_description,
                                                                  position = featuredFormData.position,
                                                                  icon_type = featuredFormData.icon_type,
                                                                  content = featuredFormData.english_content,
                                                                  primary_tag = featuredFormData.english_primary_tag,
                                                                  starred = featuredFormData.starred), "english", user)

        val futureFrenchFeatured = Featured.create(new Featured( id = frenchId,
                                                                 translated_id = englishId,
                                                                 title = featuredFormData.french_title,
                                                                 slug = frenchSlug,
                                                                 description = featuredFormData.french_description,
                                                                 position = featuredFormData.position,
                                                                 icon_type = featuredFormData.icon_type,
                                                                 content = featuredFormData.french_content,
                                                                 primary_tag = featuredFormData.french_primary_tag,
                                                                 starred = featuredFormData.starred), "french", user)

        for {
          maybeEnglishFeatured <- futureEnglishFeatured
          maybeFrenchFeatured <- futureFrenchFeatured
        }
        yield {
          Redirect("/admin#/manage/homepage").flashing("success" -> Messages("admin.featured-added"))
        }
      }
    )
  }

  def editFeatured(english_id: String, french_id: String) = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    featuredForm.bindFromRequest.fold(
      formWithErrors => {
        Future.successful(Redirect(controllers.routes.Tagger.admin).flashing("error" -> Messages("admin.badRequest")))
      },

      featuredFormData => {
        val englishSlug = {
          if (featuredFormData.english_slug == "") {
            Util.slugify(featuredFormData.english_primary_tag, featuredFormData.english_title)
          }
          else {
            featuredFormData.english_slug
          }
        }
        val frenchSlug = {
          if (featuredFormData.french_slug == "") {
            Util.slugify(featuredFormData.english_primary_tag, featuredFormData.french_title)
          }
          else {
            featuredFormData.french_slug
          }
        }

        val futureEnglishFeatured = Featured.update(new Featured( id = UUID(english_id),
                                                                  translated_id = UUID.random,
                                                                  title = featuredFormData.english_title,
                                                                  slug = englishSlug,
                                                                  description = featuredFormData.english_description,
                                                                  position = featuredFormData.position,
                                                                  icon_type = featuredFormData.icon_type,
                                                                  content = featuredFormData.english_content,
                                                                  primary_tag = featuredFormData.english_primary_tag,
                                                                  starred = featuredFormData.starred), user)

        val futureFrenchFeatured = Featured.update(new Featured(id = UUID(french_id),
                                                                translated_id = UUID.random,
                                                                title = featuredFormData.french_title,
                                                                slug = frenchSlug,
                                                                description = featuredFormData.french_description,
                                                                position = featuredFormData.position,
                                                                icon_type = featuredFormData.icon_type,
                                                                content = featuredFormData.french_content,
                                                                primary_tag = featuredFormData.french_primary_tag,
                                                                starred = featuredFormData.starred), user)

        for {
          maybeEnglishFeatured <- futureEnglishFeatured
          maybeFrenchFeatured <- futureFrenchFeatured
        }
        yield {
          Redirect("/admin#/manage/homepage").flashing("success" -> (Messages("admin.featured-edited")))
        }
      }
    )
  }

}
