package controllers.api

import play.api.mvc.Controller
import play.api.cache.Cache
import play.api.data.Form
import play.api.data.Forms._
import play.api.i18n.Messages
import play.api.libs.json.{JsObject, Json}
import play.api.libs.concurrent.Execution.Implicits._

import scala.concurrent.Future
import scala.Some

import models.{FooterItem, FooterItemFormData, Tag}
import helpers.{UUID, Util}

object FooterItems extends Controller with controllers.Secured {

  def findByLanguage(language: String) = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    FooterItem.findByLanguage(language).map { contents =>
      Ok(Json.toJson(contents.map { footerItem =>
        Json.obj("id" -> footerItem.id.string,
                 "translated_id" -> footerItem.translated_id.string,
                 "title" -> footerItem.title,
                 "description" -> footerItem.description,
                 "footer_item_position" -> footerItem.position.asInstanceOf[Int],
                 "content_type" -> footerItem.content_type,
                 "created_at" -> footerItem.createdAt,
                 "language" -> language)
      }))
    }
  }

  def footerItemForm = Form(
    mapping(
      "english-title" -> text,
      "french-title" -> text,
      "english-description" -> text,
      "french-description" -> text,
      "position" -> number,
      "english-content" -> text,
      "french-content" -> text
    )(FooterItemFormData.apply)(FooterItemFormData.unapply)
  )

  def addFooterItem = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    footerItemForm.bindFromRequest.fold(
      formWithErrors => Future.successful(Ok(Messages("admin.badRequest"))),

      footerItemFormData => {
        val englishId = UUID.random
        val frenchId = UUID.random

        val futureEnglishFooterItem = FooterItem.create(new FooterItem( id = englishId,
                                                                        translated_id = frenchId,
                                                                        title = footerItemFormData.english_title,
                                                                        slug = UUID.random.string,
                                                                        description = footerItemFormData.english_description,
                                                                        position = footerItemFormData.position,
                                                                        content = footerItemFormData.english_content), "english", user)

        val futureFrenchFooterItem = FooterItem.create(new FooterItem(id = frenchId,
                                                                      translated_id = englishId,
                                                                      title = footerItemFormData.french_title,
                                                                      slug = UUID.random.string,
                                                                      description = footerItemFormData.french_description,
                                                                      position = footerItemFormData.position,
                                                                      content = footerItemFormData.french_content), "french", user)

        for {
          maybeEnglishFooterItem <- futureEnglishFooterItem
          maybeFrenchFooterItem <- futureFrenchFooterItem
        }
        yield {
          Redirect("/admin#/manage/footerItems").flashing("success" -> Messages("admin.footer-item-added"))
        }
      }
    )
  }

  def editFooterItem(english_id: String, french_id: String) = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    footerItemForm.bindFromRequest.fold(
      formWithErrors => {
        Future.successful(Redirect(controllers.routes.Tagger.admin).flashing("error" -> Messages("admin.badRequest")))
      },

      footerItemFormData => {
        val futureEnglishFooterItem = FooterItem.update(new FooterItem( id = UUID(english_id),
                                                                  translated_id = UUID.random,
                                                                  title = footerItemFormData.english_title,
                                                                  slug = UUID.random.string,
                                                                  description = footerItemFormData.english_description,
                                                                  position = footerItemFormData.position,
                                                                  content = footerItemFormData.english_content), user)

        val futureFrenchFooterItem = FooterItem.update(new FooterItem(id = UUID(french_id),
                                                                translated_id = UUID.random,
                                                                title = footerItemFormData.french_title,
                                                                slug = UUID.random.string,
                                                                description = footerItemFormData.french_description,
                                                                position = footerItemFormData.position,
                                                                content = footerItemFormData.french_content), user)

        for {
          maybeEnglishFooterItem <- futureEnglishFooterItem
          maybeFrenchFooterItem <- futureFrenchFooterItem
        }
        yield {
          Redirect("/admin#/manage/footerItems").flashing("success" -> (Messages("admin.footer-item-edited")))
        }
      }
    )
  }

}
