package controllers.api

import play.api.mvc.Controller
import play.api.cache.Cache
import play.api.data.Form
import play.api.data.Forms._
import play.api.i18n.Messages
import play.api.libs.json.{JsObject, Json}
import play.api.libs.concurrent.Execution.Implicits._

import scala.concurrent.Future
import scala.Some

import models.{HeaderItem, HeaderItemFormData, Tag}
import helpers.{UUID, Util}

object HeaderItems extends Controller with controllers.Secured {

  def findByLanguage(language: String) = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    HeaderItem.findByLanguage(language).map { contents =>
        Ok(Json.toJson(contents.map { headerItem =>
          Json.obj("id" -> headerItem.id.string,
                   "translated_id" -> headerItem.translated_id.string,
                   "title" -> headerItem.title,
                   "description" -> headerItem.description,
                   "header_item_position" -> headerItem.position.asInstanceOf[Int],
                   "header_content" -> headerItem.content,
                   "content_type" -> headerItem.content_type,
                   "created_at" -> headerItem.createdAt,
                   "language" -> language)
        }))
    }
  }

  def headerItemForm = Form(
    mapping(
      "english-title" -> text,
      "french-title" -> text,
      "english-description" -> text,
      "french-description" -> text,
      "position" -> number,
      "english-content" -> text,
      "french-content" -> text
    )(HeaderItemFormData.apply)(HeaderItemFormData.unapply)
  )

  def setPosition(id: String) = IsAuthenticatedAsync(parse.json) { implicit user => implicit request =>
    val position = (request.body \ "position").as[Int]

    HeaderItem.setPosition(id, position).map { result =>
      result match {
        case true => Ok(Json.toJson("SUCCESS"))
        case false => Ok(Json.toJson("FAILED"))
      }
    }
  }

  def addHeaderItem = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    headerItemForm.bindFromRequest.fold(
      formWithErrors => Future.successful(Ok(Messages("admin.badRequest"))),

      headerItemFormData => {
        val englishId = UUID.random
        val frenchId = UUID.random

        val futureEnglishHeaderItem = HeaderItem.create(new HeaderItem( id = englishId,
                                                                  translated_id = frenchId,
                                                                  title = headerItemFormData.english_title,
                                                                  slug = UUID.random.string,
                                                                  description = headerItemFormData.english_description,
                                                                  position = headerItemFormData.position,
                                                                  content = headerItemFormData.english_content), "english", user)

        val futureFrenchHeaderItem = HeaderItem.create(new HeaderItem( id = frenchId,
                                                              translated_id = englishId,
                                                              title = headerItemFormData.french_title,
                                                              slug = UUID.random.string,
                                                              description = headerItemFormData.french_description,
                                                              position = headerItemFormData.position,
                                                              content = headerItemFormData.french_content), "french", user)

        for {
          maybeEnglishHeaderItem <- futureEnglishHeaderItem
          maybeFrenchHeaderItem <- futureFrenchHeaderItem
        }
        yield {
          Redirect("/admin#/manage/headerItems").flashing("success" -> Messages("admin.header-item-added"))
        }
      }
    )
  }

  def editHeaderItem(english_id: String, french_id: String) = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    headerItemForm.bindFromRequest.fold(
      formWithErrors => {
        Future.successful(Redirect(controllers.routes.Tagger.admin).flashing("error" -> Messages("admin.badRequest")))
      },

      headerItemFormData => {
        val futureEnglishHeaderItem = HeaderItem.update(new HeaderItem( id = UUID(english_id),
                                                                  translated_id = UUID.random,
                                                                  title = headerItemFormData.english_title,
                                                                  slug = UUID.random.string,
                                                                  description = headerItemFormData.english_description,
                                                                  position = headerItemFormData.position,
                                                                  content = headerItemFormData.english_content), user)

        val futureFrenchHeaderItem = HeaderItem.update(new HeaderItem(id = UUID(french_id),
                                                                translated_id = UUID.random,
                                                                title = headerItemFormData.french_title,
                                                                slug = UUID.random.string,
                                                                description = headerItemFormData.french_description,
                                                                position = headerItemFormData.position,
                                                                content = headerItemFormData.french_content), user)

        for {
          maybeEnglishHeaderItem <- futureEnglishHeaderItem
          maybeFrenchHeaderItem <- futureFrenchHeaderItem
        }
        yield {
          Redirect("/admin#/manage/headerItems").flashing("success" -> (Messages("admin.header-item-edited")))
        }
      }
    )
  }

}
