package controllers.api

import play.api.mvc.Controller
import play.api.cache.Cache
import play.api.data.Form
import play.api.data.Forms._
import play.api.i18n.Messages
import play.api.libs.json.{JsObject, Json}
import play.api.libs.concurrent.Execution.Implicits._

import scala.concurrent.Future
import scala.Some

import models.{Link, LinkFormData, Tag}
import helpers.{UUID, Util}

object Links extends Controller with controllers.Secured {

  def linkForm = Form(
    mapping(
      "english-title" -> text,
      "french-title" -> text,
      "english-slug" -> text,
      "french-slug" -> text,
      "english-description" -> text,
      "french-description" -> text,
      "english_link_url" -> text,
      "french_link_url" -> text,
      "english-primary-tag" -> text,
      "french-primary-tag" -> text,
      "english-tags" -> text,
      "french-tags" -> text,
      "starred" -> number
    )(LinkFormData.apply)(LinkFormData.unapply)
  )

  def addLink = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    linkForm.bindFromRequest.fold(
      formWithErrors => Future.successful(Ok(Messages("admin.badRequest"))),

      linkFormData => {
        val englishId = UUID.random
        val frenchId = UUID.random
        val englishPrimaryTag = linkFormData.english_tags.split(",").distinct(0).trim
        val frenchPrimaryTag = linkFormData.french_tags.split(",").distinct(0).trim
        val englishSlug = {
          if (linkFormData.english_slug == "") {
            Util.slugify(englishPrimaryTag, linkFormData.english_title)
          }
          else {
            Util.slugify(englishPrimaryTag, linkFormData.english_slug)
          }
        }
        val frenchSlug = {
          if (linkFormData.french_slug == "") {
            Util.slugify(frenchPrimaryTag, linkFormData.french_title)
          }
          else {
            Util.slugify(frenchPrimaryTag, linkFormData.french_slug)
          }
        }

        val futureEnglishLink = Link.create(new Link( id = englishId,
                                                      translated_id = frenchId,
                                                      title = linkFormData.english_title,
                                                      slug = englishSlug,
                                                      description = linkFormData.english_description,
                                                      primary_tag = englishPrimaryTag,
                                                      starred = linkFormData.starred,
                                                      link_url = linkFormData.english_link_url), "english", user)

        val futureFrenchLink = Link.create(new Link(  id = frenchId,
                                                      translated_id = englishId,
                                                      title = linkFormData.french_title,
                                                      slug = frenchSlug,
                                                      description = linkFormData.french_description,
                                                      primary_tag = frenchPrimaryTag,
                                                      starred = linkFormData.starred,
                                                      link_url = linkFormData.french_link_url), "french", user)

        for {
          maybeEnglishLink <- futureEnglishLink
          maybeFrenchLink <- futureFrenchLink
        }
        yield {
          (maybeEnglishLink, maybeFrenchLink) match {
            case (Some(english), Some(french)) => {
              Tag.tagContent(linkFormData.english_tags, english.id.bytes, "english")
              Tag.tagContent(linkFormData.french_tags, french.id.bytes, "french")
            }
            case _ => {}
          }
          Redirect(controllers.routes.Tagger.admin).flashing("success" -> Messages("admin.link-added"))
        }
      }
    )
  }

  def editLink(english_id: String, french_id: String) = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    linkForm.bindFromRequest.fold(
      formWithErrors => Future.successful(Redirect(controllers.routes.Tagger.admin).flashing("error" -> Messages("admin.badRequest"))),

      linkFormData => {
        val englishSlug = {
          if (linkFormData.english_slug == "") {
            Util.slugify(linkFormData.english_primary_tag, linkFormData.english_title)
          }
          else {
            linkFormData.english_slug
          }
        }
        val frenchSlug = {
          if (linkFormData.french_slug == "") {
            Util.slugify(linkFormData.french_primary_tag, linkFormData.french_title)
          }
          else {
            linkFormData.french_slug
          }
        }

        val futureEnglishLink = Link.update(new Link( id = UUID(english_id),
                                                      translated_id = UUID.random,
                                                      title = linkFormData.english_title,
                                                      slug = englishSlug,
                                                      description = linkFormData.english_description,
                                                      primary_tag = linkFormData.english_primary_tag,
                                                      starred = linkFormData.starred,
                                                      link_url = linkFormData.english_link_url), user)

        val futureFrenchLink = Link.update(new Link(id = UUID(french_id),
                                                    translated_id = UUID.random,
                                                    title = linkFormData.french_title,
                                                    slug = frenchSlug,
                                                    description = linkFormData.french_description,
                                                    primary_tag = linkFormData.french_primary_tag,
                                                    starred = linkFormData.starred,
                                                    link_url = linkFormData.french_link_url), user)

        for {
          maybeEnglishLink <- futureEnglishLink
          maybeFrenchLink <- futureFrenchLink
        }
        yield {
          (maybeEnglishLink, maybeFrenchLink) match {
            case (Some(english), Some(french)) => {
              Tag.tagContent(linkFormData.english_tags, UUID(english_id).bytes, "english")
              Tag.tagContent(linkFormData.french_tags, UUID(french_id).bytes, "french")
            }
            case _ => {}
          }
          Redirect("/admin#/edit/tagged/content").flashing("success" -> (Messages("admin.link-edited")))
        }
      }
    )
  }

}
