package controllers.api

import play.api.mvc.Controller
import play.api.cache.Cache
import play.api.data.Form
import play.api.data.Forms._
import play.api.i18n.Messages
import play.api.libs.json.{JsObject, Json}
import play.api.libs.concurrent.Execution.Implicits._
import play.api.Logger

import scala.concurrent.Future
import scala.Some

import models.MenuItem
import helpers.UUID

object MenuItems extends Controller with controllers.Secured {

  def findAllParents(language: String) = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    MenuItem.findAllParents(language).map { menuItems =>
      Ok(Json.toJson(menuItems.map { menuItem => Json.toJson(menuItem) }))
    }
  }

  def findByParentId(parentId: String, language: String) = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    MenuItem.findByParentId(parentId, language).map { menuItems =>
      Ok(Json.toJson(menuItems.map { menuItem => Json.toJson(menuItem) }))
    }
  }

  def setPosition(id: String) = IsAuthenticatedAsync(parse.json) { implicit user => implicit request =>
    val position = (request.body \ "position").as[Int]

    MenuItem.setPosition(id, position).map { result => Ok(Json.toJson(result)) }
  }

  def create = IsAuthenticatedAsync(parse.json) { implicit user => implicit request =>
    val id = UUID.random.string
    val parentId = (request.body \ "parentId").asOpt[String]
    val tagLineId = (request.body \ "tagLineId").asOpt[String]
    val contentId =  (request.body \ "contentId").asOpt[String]
    val title = (request.body \ "title").as[String]
    val description = (request.body \ "description").as[String]
    val longDescription = (request.body \ "longDescription").as[String]
    val published = (request.body \ "published").as[Int]
    val fLargestPosition = MenuItem.getLargestPosition

    for {
      cLargestPosition <- fLargestPosition
      cMenuItem <- MenuItem.create(new MenuItem(  id = id,
                                                  parentId = parentId,
                                                  tagLineId = tagLineId,
                                                  contentId = contentId,
                                                  title = title,
                                                  description = description,
                                                  longDescription = longDescription,
                                                  published = published,
                                                  language = "french",
                                                  position = cLargestPosition.getOrElse{0} + 1))
    } yield {
      Ok(Json.toJson(cMenuItem))
    }
  }

  def save = IsAuthenticatedAsync(parse.json) { implicit user => implicit request =>
    MenuItem.update(request.body.as[MenuItem]).map { menuItems =>
      Ok(Json.toJson(menuItems.map { menuItem => Json.toJson(menuItem) }))
    }
  }

  def deleteMenuItem(id: String, deleteSubMenus: String) = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    deleteSubMenus match {
      case "true" => MenuItem.deleteCascade(id).map { result => Ok(Json.toJson(result)) }
      case "false" => MenuItem.delete(id).map { result => Ok(Json.toJson(result)) }
      case _ => MenuItem.deleteCascade(id).map { result => Ok(Json.toJson(result)) }
    }
  }

}
