package controllers.api

import play.api.mvc.Controller
import play.api.cache.Cache
import play.api.data.Form
import play.api.data.Forms._
import play.api.i18n.Messages
import play.api.libs.json.{JsObject, Json}
import play.api.libs.concurrent.Execution.Implicits._

import scala.concurrent.Future
import scala.Some

import models._
import helpers.{UUID, Util}

object SelectFeatureds extends Controller with controllers.Secured {

  def findByLanguage(language: String) = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    SelectFeatured.findByLanguage(language).map { contents =>
        Ok(Json.toJson(contents.map { selectFeatured =>
          Json.obj("id" -> selectFeatured.id.string,
                   "translated_id" -> selectFeatured.translated_id.string,
                   "title" -> selectFeatured.title,
                   "description" -> selectFeatured.description,
                   "select_featured_href_slug" -> selectFeatured.href_slug,
                   "select_featured_position" -> selectFeatured.position.asInstanceOf[Int],
                   "select_featured_icon_type" -> selectFeatured.icon_type,
                   "content_type" -> selectFeatured.content_type,
                   "primary_tag" -> selectFeatured.primary_tag,
                   "starred" -> selectFeatured.starred.asInstanceOf[Int],
                   "created_at" -> selectFeatured.createdAt,
                   "language" -> language)
        }))
    }
  }

  def selectFeaturedForm = Form(
    mapping(
      "english-title" -> text,
      "french-title" -> text,
      "english-description" -> text,
      "french-description" -> text,
      "href-id" -> text,
      "position" -> number,
      "icon_type" -> text,
      "english-primary-tag" -> text,
      "french-primary-tag" -> text,
      "starred" -> number
    )(SelectFeaturedFormData.apply)(SelectFeaturedFormData.unapply)
  )

  def addSelectFeatured = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    selectFeaturedForm.bindFromRequest.fold(
      formWithErrors => Future.successful(Ok(Messages("admin.badRequest"))),

      selectFeaturedFormData => {
        val englishId = UUID.random
        val frenchId = UUID.random

        val futureEnglishSelectFeatured = SelectFeatured.create(new SelectFeatured( id = englishId,
                                                                  translated_id = frenchId,
                                                                  title = selectFeaturedFormData.english_title,
                                                                  slug = UUID.random.string,
                                                                  description = selectFeaturedFormData.english_description,
                                                                  href_slug = selectFeaturedFormData.href_slug,
                                                                  position = selectFeaturedFormData.position,
                                                                  icon_type = selectFeaturedFormData.icon_type,
                                                                  primary_tag = selectFeaturedFormData.english_primary_tag,
                                                                  starred = selectFeaturedFormData.starred), "english", user)

        val futureFrenchSelectFeatured = SelectFeatured.create(new SelectFeatured( id = frenchId,
                                                                 translated_id = englishId,
                                                                 title = selectFeaturedFormData.french_title,
                                                                 slug = UUID.random.string,
                                                                 description = selectFeaturedFormData.french_description,
                                                                 href_slug = selectFeaturedFormData.href_slug,
                                                                 position = selectFeaturedFormData.position,
                                                                 icon_type = selectFeaturedFormData.icon_type,
                                                                 primary_tag = selectFeaturedFormData.french_primary_tag,
                                                                 starred = selectFeaturedFormData.starred), "french", user)

        for {
          maybeEnglishSelectFeatured <- futureEnglishSelectFeatured
          maybeFrenchSelectFeatured <- futureFrenchSelectFeatured
        }
        yield {
          Redirect("/admin#/manage/selectFeatured").flashing("success" -> Messages("admin.select-featured-added"))
        }
      }
    )
  }

  def editSelectFeatured(english_id: String, french_id: String) = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    selectFeaturedForm.bindFromRequest.fold(
      formWithErrors => {
        Future.successful(Redirect(controllers.routes.Tagger.admin).flashing("error" -> Messages("admin.badRequest")))
      },

      selectFeaturedFormData => {
        val englishSlug = Util.slugify(selectFeaturedFormData.english_primary_tag, selectFeaturedFormData.english_title)
        val frenchSlug = Util.slugify(selectFeaturedFormData.english_primary_tag, selectFeaturedFormData.french_title)

        val futureEnglishSelectFeatured = SelectFeatured.update(new SelectFeatured( id = UUID(english_id),
                                                                  translated_id = UUID.random,
                                                                  title = selectFeaturedFormData.english_title,
                                                                  slug = englishSlug,
                                                                  description = selectFeaturedFormData.english_description,
                                                                  href_slug = selectFeaturedFormData.href_slug,
                                                                  position = selectFeaturedFormData.position,
                                                                  icon_type = selectFeaturedFormData.icon_type,
                                                                  primary_tag = selectFeaturedFormData.english_primary_tag,
                                                                  starred = selectFeaturedFormData.starred), user)

        val futureFrenchFeatured = SelectFeatured.update(new SelectFeatured(id = UUID(french_id),
                                                                translated_id = UUID.random,
                                                                title = selectFeaturedFormData.french_title,
                                                                slug = frenchSlug,
                                                                description = selectFeaturedFormData.french_description,
                                                                href_slug = selectFeaturedFormData.href_slug,
                                                                position = selectFeaturedFormData.position,
                                                                icon_type = selectFeaturedFormData.icon_type,
                                                                primary_tag = selectFeaturedFormData.french_primary_tag,
                                                                starred = selectFeaturedFormData.starred), user)
        Future(Redirect("/admin#/manage/selectFeatured").flashing("success" -> (Messages("admin.featured-edited"))))
      }
    )
  }

}
