package controllers.api

import play.api.mvc.Controller
import play.api.cache.Cache
import play.api.data.Form
import play.api.data.Forms._
import play.api.i18n.Messages
import play.api.libs.json.{JsObject, Json}
import play.api.libs.concurrent.Execution.Implicits._
import play.api.Logger

import scala.concurrent.Future
import scala.Some

import models.{SiteWord, SiteWordFormData, Tag}
import helpers.{UUID, Util, JsMessages}

object SiteWords extends Controller with controllers.Secured {

  def findByLanguage(language: String) = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    SiteWord.findByLanguage(language).map { contents =>
      Ok(Json.toJson(contents.map { siteWord =>
        Json.obj("id" -> siteWord.id.string,
                 "translated_id" -> siteWord.translated_id.string,
                 "title" -> siteWord.title,
                 "description" -> siteWord.description,
                 "site_word_href_id" -> siteWord.href_id.string,
                 "site_word_position" -> siteWord.position.asInstanceOf[Int],
                 "content_type" -> siteWord.content_type,
                 "created_at" -> siteWord.createdAt,
                 "language" -> language)
      }))
    }
  }

  def siteWordForm = Form(
    mapping(
      "english-title" -> text,
      "french-title" -> text,
      "english-description" -> text,
      "french-description" -> text,
      "href-id" -> text,
      "position" -> number
    )(SiteWordFormData.apply)(SiteWordFormData.unapply)
  )

  def setPosition(id: String) = IsAuthenticatedAsync(parse.json) { implicit user => implicit request =>
    val position = (request.body \ "position").as[Int]

    SiteWord.setPosition(id, position).map { result =>
      result match {
        case true => Ok(Json.toJson("SUCCESS"))
        case false => Ok(Json.toJson("FAILED"))
      }
    }
  }

  def addSiteWord = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    siteWordForm.bindFromRequest.fold(
      formWithErrors => Future.successful(Ok(Messages("admin.badRequest"))),

      siteWordFormData => {
        val englishId = UUID.random
        val frenchId = UUID.random

        val futureEnglishSiteWord = SiteWord.create(new SiteWord( id = englishId,
                                                                  translated_id = frenchId,
                                                                  title = siteWordFormData.english_title,
                                                                  slug = UUID.random.string,
                                                                  description = siteWordFormData.english_description,
                                                                  href_id = UUID(siteWordFormData.href_id),
                                                                  position = siteWordFormData.position), "english", user)

        val futureFrenchSiteWord = SiteWord.create(new SiteWord( id = frenchId,
                                                                 translated_id = englishId,
                                                                 title = siteWordFormData.french_title,
                                                                 slug = UUID.random.string,
                                                                 description = siteWordFormData.french_description,
                                                                 href_id = UUID(siteWordFormData.href_id),
                                                                 position = siteWordFormData.position), "french", user)

        for {
          maybeEnglishSiteWord <- futureEnglishSiteWord
          maybeFrenchSiteWord <- futureFrenchSiteWord
        }
        yield {
          Redirect("/admin#/manage/siteWords").flashing("success" -> Messages("admin.site-word-added"))
        }
      }
    )
  }

  def editSiteWord(english_id: String, french_id: String) = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    siteWordForm.bindFromRequest.fold(
      formWithErrors => {
        Future.successful(Redirect(controllers.routes.Tagger.admin).flashing("error" -> Messages("admin.badRequest")))
      },

      siteWordFormData => {
        val futureEnglishSiteWord = SiteWord.update(new SiteWord( id = UUID(english_id),
                                                                  translated_id = UUID.random,
                                                                  title = siteWordFormData.english_title,
                                                                  slug = UUID.random.string,
                                                                  description = siteWordFormData.english_description,
                                                                  href_id = UUID(siteWordFormData.href_id),
                                                                  position = siteWordFormData.position), user)

        val futureFrenchSiteWord = SiteWord.update(new SiteWord(id = UUID(french_id),
                                                                translated_id = UUID.random,
                                                                title = siteWordFormData.french_title,
                                                                slug = UUID.random.string,
                                                                description = siteWordFormData.french_description,
                                                                href_id = UUID(siteWordFormData.href_id),
                                                                position = siteWordFormData.position), user)

        for {
          maybeEnglishSiteWord <- futureEnglishSiteWord
          maybeFrenchSiteWord <- futureFrenchSiteWord
        }
        yield {
          Redirect("/admin#/manage/siteWords").flashing("success" -> (Messages("admin.site-word-edited")))
        }
      }
    )
  }
}
