package controllers.api

import play.api.Play
import play.api.mvc.Controller
import play.api.cache.Cache
import play.api.data.Form
import play.api.data.Forms._
import play.api.i18n.Messages
import play.api.libs.json.{JsObject, Json}
import play.api.libs.concurrent.Execution.Implicits._

import scala.concurrent.duration._
import scala.concurrent.{Future, Await}
import scala.Some

import util.Failure

import java.io.File

import models._
import helpers.{UUID, Util, JsMessages}

object Sites extends Controller with controllers.Secured {

  def siteForm = Form(
    mapping(
      "english-title" -> text,
      "french-title" -> text,
      "english-description" -> text,
      "french-description" -> text,
      "english-content" -> text,
      "french-content" -> text,
      "english-site-url" -> text,
      "french-site-url" -> text,
      "english-tags" -> text,
      "french-tags" -> text,
      "starred" -> number
    )(SiteFormData.apply)(SiteFormData.unapply)
  )

  def addSite = IsAuthenticatedAsync(parse.multipartFormData) { implicit user => implicit request =>
    siteForm.bindFromRequest.fold(
      formWithErrors => Future.successful(Ok(Messages("admin.badRequest"))),

      siteFormData => {
        val englishId = UUID.random
        val frenchId = UUID.random

        var siteLogo = UUID.random.string

        request.body.file("site-logo").map {
          picture => {
            picture.ref.moveTo(new File(Play.current.configuration.getString("uploads.outFile").get + siteLogo))
          }
        }.getOrElse {
          siteLogo = "no-image.png"
        }

        val englishPrimaryTag = siteFormData.english_tags.split(",").distinct(0).trim
        val frenchPrimaryTag = siteFormData.french_tags.split(",").distinct(0).trim
        val englishSlug = UUID.random.string
        val frenchSlug = UUID.random.string
        val englishDescription = {
          if (siteFormData.english_description.length > 150)
            siteFormData.english_description.substring(0, 150) + "..."
          else siteFormData.english_description
        }
        val frenchDescription = {
          if (siteFormData.french_description.length > 150)
            siteFormData.french_description.substring(0, 150) + "..."
          else siteFormData.french_description
        }

        val futureEnglishSite = Site.create(new Site( id = englishId,
                                                      translated_id = frenchId,
                                                      title = siteFormData.english_title,
                                                      slug = englishSlug,
                                                      description = englishDescription,
                                                      primary_tag = englishPrimaryTag,
                                                      starred = siteFormData.starred,
                                                      site_logo = siteLogo,
                                                      content = siteFormData.english_content,
                                                      site_url = siteFormData.english_site_url), "english", user)

        val futureFrenchSite = Site.create(new Site(  id = frenchId,
                                                      translated_id = englishId,
                                                      title = siteFormData.french_title,
                                                      slug = frenchSlug,
                                                      description = frenchDescription,
                                                      primary_tag = frenchPrimaryTag,
                                                      starred = siteFormData.starred,
                                                      site_logo = siteLogo,
                                                      content = siteFormData.french_content,
                                                      site_url = siteFormData.french_site_url), "french", user)

        for {
          maybeEnglishSite <- futureEnglishSite
          maybeFrenchSite <- futureFrenchSite
        }
        yield {
          (maybeEnglishSite, maybeFrenchSite) match {
            case (Some(english), Some(french)) => {
              Tag.tagContent(siteFormData.english_tags, english.id.bytes, "english")
              Tag.tagContent(siteFormData.french_tags, french.id.bytes, "french")
            }
            case _ => {}
          }
          Redirect(controllers.routes.Tagger.admin).flashing("success" -> Messages("admin.site-added"))
        }
      }
    )
  }

  def editSite(english_id: String, french_id: String) = IsAuthenticatedAsync(parse.multipartFormData) { implicit user => implicit request =>
    siteForm.bindFromRequest.fold(
      formWithErrors => Future.successful(Redirect(controllers.routes.Tagger.admin).flashing("error" -> Messages("admin.badRequest"))),

      siteFormData => {
        val site = Await.result(ContentModel.find(UUID(english_id)), 5 seconds).get.asInstanceOf[Site]
        var siteLogo = site.site_logo

        request.body.file("site-logo").map {
          picture => {
            val logoName = picture.filename
            val logoType = picture.contentType.get

            // Delete the old logo unless it's the 'no-image' file.
            val oldSiteLogo = new File(Play.current.configuration.getString("uploads.outFile").get + site.site_logo)
            if (site.site_logo != "no-image.png") {
              oldSiteLogo.delete
            }

            siteLogo = UUID.random.string
            picture.ref.moveTo(new File(Play.current.configuration.getString("uploads.outFile").get + siteLogo))
          }
        }.getOrElse {
          siteLogo = site.site_logo
        }

        val englishPrimaryTag = siteFormData.english_tags.split(",").distinct(0).trim
        val frenchPrimaryTag = siteFormData.french_tags.split(",").distinct(0).trim
        val englishSlug = UUID.random.string
        val frenchSlug = UUID.random.string
        val englishDescription = {
          if (siteFormData.english_description.length > 150)
            siteFormData.english_description.substring(0, 150) + "..."
          else siteFormData.english_description
        }
        val frenchDescription = {
          if (siteFormData.french_description.length > 150)
            siteFormData.french_description.substring(0, 150) + "..."
          else siteFormData.french_description
        }

        val futureEnglishSite = Site.update(new Site( id = UUID(english_id),
                                                      translated_id = UUID.random,
                                                      title = siteFormData.english_title,
                                                      slug = englishSlug,
                                                      description = englishDescription,
                                                      primary_tag = englishPrimaryTag,
                                                      starred = siteFormData.starred,
                                                      site_logo = siteLogo,
                                                      content = siteFormData.english_content,
                                                      site_url = siteFormData.english_site_url), user)

        val futureFrenchSite = Site.update(new Site(id = UUID(french_id),
                                                    translated_id = UUID.random,
                                                    title = siteFormData.french_title,
                                                    slug = frenchSlug,
                                                    description = frenchDescription,
                                                    primary_tag = frenchPrimaryTag,
                                                    starred = siteFormData.starred,
                                                    site_logo = siteLogo,
                                                    content = siteFormData.french_content,
                                                    site_url = siteFormData.french_site_url), user)

        for {
          maybeEnglishSite <- futureEnglishSite
          maybeFrenchSite <- futureFrenchSite
        }
        yield {
          (maybeEnglishSite, maybeFrenchSite) match {
            case (Some(english), Some(french)) => {
              Tag.tagContent(siteFormData.english_tags, english.id.bytes, "english")
              Tag.tagContent(siteFormData.french_tags, french.id.bytes, "french")
            }
            case _ => {}
          }
          Redirect(controllers.routes.Tagger.admin).flashing("success" -> Messages("admin.site-edited"))
        }
      }
    )
  }
}
