package controllers.api

import play.api.Play
import play.api.mvc.Controller
import play.api.cache.Cache
import play.api.data.Form
import play.api.data.Forms._
import play.api.i18n.Messages
import play.api.libs.json.{JsObject, Json}
import play.api.libs.concurrent.Execution.Implicits._

import scala.concurrent.duration._
import scala.concurrent.{Future, Await}
import scala.Some

import util.Failure

import java.io.File

import models._
import helpers.{UUID, Util, JsMessages}

object Slides extends Controller with controllers.Secured {

  def findByLanguage(language: String) = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    Slide.findByLanguage(language).map { contents =>
        Ok(Json.toJson(contents.map { slide =>
          Json.obj("id" -> slide.id.string,
                   "translated_id" -> slide.translated_id.string,
                   "title" -> slide.title,
                   "slug" -> slide.slug,
                   "description" -> slide.description,
                   "slide_position" -> slide.position.asInstanceOf[Int],
                   "slide_background" -> slide.slide_background,
                   "header_colour" -> slide.header_colour,
                   "content_type" -> slide.content_type,
                   "primary_tag" -> slide.primary_tag,
                   "created_at" -> slide.createdAt,
                   "language" -> language)
        }))
    }
  }

  def slideForm = Form(
    mapping(
      "english-title" -> text,
      "french-title" -> text,
      "english-description" -> text,
      "french-description" -> text,
      "position" -> number,
      "english-slide-url" -> text,
      "french-slide-url" -> text,
      "header-colour" -> text
    )(SlideFormData.apply)(SlideFormData.unapply)
  )

  def addSlide = IsAuthenticatedAsync(parse.multipartFormData) { implicit user => implicit request =>
    slideForm.bindFromRequest.fold(
      formWithErrors => Future.successful(Ok(Messages("admin.badRequest"))),

      slideFormData => {
        val englishId = UUID.random
        val frenchId = UUID.random

        var slideBackground = ""

        request.body.file("slide-background").map {
          picture => {
            val backgroundName = picture.filename.split("\\.").head
            val backgroundType = picture.filename.split("\\.").last

            slideBackground = backgroundName + UUID.random.string + "." + backgroundType

            picture.ref.moveTo(new File(Play.current.configuration.getString("uploads.outFile").get + slideBackground))
          }
        }.getOrElse {
          slideBackground = "no-image.png"
        }

        val futureEnglishSlide = Slide.create(new Slide(id = englishId,
                                                        translated_id = frenchId,
                                                        title = slideFormData.english_title,
                                                        slug = UUID.random.string,
                                                        description = slideFormData.english_description,
                                                        position = slideFormData.position,
                                                        slide_background = slideBackground,
                                                        slide_url = slideFormData.english_slide_url,
                                                        header_colour = slideFormData.header_colour), "english", user)

        val futureFrenchSlide = Slide.create(new Slide(id = frenchId,
                                                       translated_id = englishId,
                                                       title = slideFormData.french_title,
                                                       slug = UUID.random.string,
                                                       description = slideFormData.french_description,
                                                       position = slideFormData.position,
                                                       slide_background = slideBackground,
                                                       slide_url = slideFormData.french_slide_url,
                                                       header_colour = slideFormData.header_colour), "french", user)

        for {
          maybeEnglishSlide <- futureEnglishSlide
          maybeFrenchSlide <- futureFrenchSlide
        }
        yield {
          Redirect(controllers.routes.Tagger.admin).flashing("success" -> Messages("admin.slide-added"))
        }
      }
    )
  }

  def editSlide(english_id: String, french_id: String) = IsAuthenticatedAsync(parse.multipartFormData) { implicit user => implicit request =>
    slideForm.bindFromRequest.fold(
      formWithErrors => Future.successful(Redirect(controllers.routes.Tagger.admin).flashing("error" -> Messages("admin.badRequest"))),

      slideFormData => {
        var slide = Await.result(ContentModel.find(UUID(english_id)), 5 seconds).get.asInstanceOf[Slide]
        var slideBackground = slide.slide_background

        request.body.file("slide-background").map {
          picture => {
            val backgroundName = picture.filename.split("\\.").head

            // Delete the old background unless it's the 'no-image' file.
            val oldSlideBackground = new File(Play.current.configuration.getString("uploads.outFile").get + slide.slide_background)
            if (slide.slide_background != "no-image.png") {
              oldSlideBackground.delete
            }

            slideBackground = backgroundName + UUID.random.string + "." + request.body.file("slide-background").get.filename.split("\\.").last
            picture.ref.moveTo(new File(Play.current.configuration.getString("uploads.outFile").get + slideBackground))
          }
        }.getOrElse {
          slideBackground = slide.slide_background
        }

        var futureEnglishSlide = Slide.update(new Slide( id = UUID(english_id),
                                                      translated_id = UUID.random,
                                                      title = slideFormData.english_title,
                                                      description = slideFormData.english_description,
                                                      position = slideFormData.position,
                                                      slug = UUID.random.string,
                                                      slide_background = slideBackground,
                                                      slide_url = slideFormData.english_slide_url,
                                                      header_colour = slideFormData.header_colour), user)

        var futureFrenchSlide = Slide.update(new Slide(id = UUID(french_id),
                                                    translated_id = UUID.random,
                                                    title = slideFormData.french_title,
                                                    description = slideFormData.french_description,
                                                    position = slideFormData.position,
                                                    slug = UUID.random.string,
                                                    slide_background = slideBackground,
                                                    slide_url = slideFormData.french_slide_url,
                                                    header_colour = slideFormData.header_colour), user)

        for {
          maybeEnglishSlide <- futureEnglishSlide
          maybeFrenchSlide <- futureFrenchSlide
        }
        yield {
          Redirect(controllers.routes.Tagger.admin).flashing("success" -> Messages("admin.slide-edited"))
        }
      }
    )
  }
}
