package controllers.api

import play.api.mvc.Controller
import play.api.cache.Cache
import play.api.data.Form
import play.api.data.Forms._
import play.api.i18n.Messages
import play.api.libs.json.{JsObject, Json}
import play.api.libs.concurrent.Execution.Implicits._

import scala.concurrent.duration._
import scala.concurrent.{Future, Await}
import scala.Some

import models.{TagLine, TagLineFormData, Tag}
import helpers.{UUID, Util, JsMessages}

object TagLines extends Controller with controllers.Secured {

  def findByLanguage(language: String) = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    TagLine.findByLanguage(language).map { contents =>
        Ok(Json.toJson(contents.map { tagLine =>
          val futureTags = Tag.findByTagged(tagLine.id.bytes)
          val createdTags = Await.result(futureTags, 5 seconds)
          var tags = Seq[String]()

          for(tag <- createdTags) { tags = (tags :+ tag.name) }

          Json.obj("id" -> tagLine.id.string,
                   "translated_id" -> tagLine.translated_id.string,
                   "title" -> tagLine.title,
                   "description" -> tagLine.description,
                   "tag_line_position" -> tagLine.position.asInstanceOf[Int],
                   "content_type" -> tagLine.content_type,
                   "created_at" -> tagLine.createdAt,
                   "tags" -> tags.mkString(","))
        }))
    }
  }

  def tagLineForm = Form(
    mapping(
      "english-title" -> text,
      "french-title" -> text,
      "english-description" -> text,
      "french-description" -> text,
      "position" -> number,
      "english-tags" -> text,
      "french-tags" -> text
    )(TagLineFormData.apply)(TagLineFormData.unapply)
  )

  def setPosition(id: String) = IsAuthenticatedAsync(parse.json) { implicit user => implicit request =>
    val position = (request.body \ "position").as[Int]

    TagLine.setPosition(id, position).map { result =>
      result match {
        case true => Ok(Json.toJson("SUCCESS"))
        case false => Ok(Json.toJson("FAILED"))
      }
    }
  }

  def addTagLine = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    tagLineForm.bindFromRequest.fold(
      formWithErrors => {
        Future.successful(Ok(Messages("admin.badRequest")))
      },

      tagLineFormData => {
        val englishId = UUID.random
        val frenchId = UUID.random

        val futureEnglishTagLine = TagLine.create(new TagLine( id = englishId,
                                                               translated_id = frenchId,
                                                               title = tagLineFormData.english_title,
                                                               slug = UUID.random.string,
                                                               description = tagLineFormData.english_description,
                                                               position = tagLineFormData.position), "english", user)

        val futureFrenchTagLine = TagLine.create(new TagLine( id = frenchId,
                                                              translated_id = englishId,
                                                              title = tagLineFormData.french_title,
                                                              slug = UUID.random.string,
                                                              description = tagLineFormData.french_description,
                                                              position = tagLineFormData.position), "french", user)

        for {
          maybeEnglishTagLine <- futureEnglishTagLine
          maybeFrenchTagLine <- futureFrenchTagLine
        }
        yield {
          (maybeEnglishTagLine, maybeFrenchTagLine) match {
            case (Some(english), Some(french)) => {
              Tag.tagContent(tagLineFormData.english_tags, english.id.bytes, "english")
              Tag.tagContent(tagLineFormData.french_tags, french.id.bytes, "french")
            }
            case _ => {}
          }
          Redirect("/admin#/manage/tagLines").flashing("success" -> Messages("admin.tag-line-added"))
        }
      }
    )
  }

  def editTagLine(english_id: String, french_id: String) = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    tagLineForm.bindFromRequest.fold(
      formWithErrors => Future.successful(Redirect(controllers.routes.Tagger.admin).flashing("error" -> Messages("admin.badRequest"))),

      tagLineFormData => {
        val futureEnglishTagLine = TagLine.update(new TagLine( id = UUID(english_id),
                                                               translated_id = UUID.random,
                                                               title = tagLineFormData.english_title,
                                                               slug = UUID.random.string,
                                                               description = tagLineFormData.english_description,
                                                               position = tagLineFormData.position), user)

        val futureFrenchTagLine = TagLine.update(new TagLine(id = UUID(french_id),
                                                             translated_id = UUID.random,
                                                             title = tagLineFormData.french_title,
                                                             slug = UUID.random.string,
                                                             description = tagLineFormData.french_description,
                                                             position = tagLineFormData.position), user)

        for {
          maybeEnglishTagLine <- futureEnglishTagLine
          maybeFrenchTagLine <- futureFrenchTagLine
        }
        yield {
          (maybeEnglishTagLine, maybeFrenchTagLine) match {
            case (Some(english), Some(french)) => {
              Tag.tagContent(tagLineFormData.english_tags, UUID(english_id).bytes, "english")
              Tag.tagContent(tagLineFormData.french_tags, UUID(french_id).bytes, "french")
            }
            case _ => {}
          }
          Redirect("/admin#/manage/tagLines").flashing("success" -> (Messages("admin.tag-line-edited")))
        }
      }
    )
  }

}
