package controllers.api

import play.api.mvc.Controller
import play.api.cache.Cache
import play.api.data.Form
import play.api.data.Forms._
import play.api.i18n.Messages
import play.api.libs.json.{JsObject, Json}
import play.api.libs.concurrent.Execution.Implicits._

import scala.concurrent.duration._
import scala.concurrent.{Future, Await}
import scala.Some

import models.{Tag, TagFormData, ContentModel}
import helpers.{UUID, Util, JsMessages}

object Tags extends Controller with controllers.Secured {

  def tagForm = Form(
    mapping(
      "name" -> text
    )(TagFormData.apply)(TagFormData.unapply)
  )

  def findByLanguage(language: String) = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    Tag.findByLanguage(language).map { tags =>
      Ok(Json.toJson(tags.map { tag =>

        Json.obj("id" -> tag.id.string,
                 "name"  -> tag.name,
                 "count" -> tag.count)
      }))
    }
  }

  def findByTagged(content_id: String) = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    Tag.findByTagged(UUID(content_id).bytes).map { tags =>
      Ok(Json.toJson(tags.map { tag =>

        Json.obj( "id" -> tag.id.string,
                  "name" -> tag.name,
                  "count" -> tag.count)
      }))
    }
  }

  def findByManyTagged(content_ids: String) = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    Tag.findByManyTagged(content_ids).map { tags =>
      Ok(Json.toJson(tags.map { tag =>

        Json.obj( "id" -> tag.id.string,
                  "name" -> tag.name,
                  "count" -> tag.count)
      }))
    }
  }

  def findByTags(tags: String, language: String) = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    Tag.findByTags(tags, language).map { tags =>
      Ok(Json.toJson(tags.map { tag =>

        Json.obj( "id" -> tag.id.string,
                  "name" -> tag.name,
                  "count" -> tag.count)
      }))
    }
  }

  def findById(id: String) = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    Tag.find(UUID(id)).map { tag =>
      Ok(Json.toJson(tag.map { tag =>

        Json.obj( "id" -> tag.id.string,
                  "name" -> tag.name,
                  "count" -> tag.count)
      }))
    }
  }

  def editTag(id: String) = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    tagForm.bindFromRequest.fold(
      formWithErrors => Future.successful(Redirect(controllers.routes.Tagger.admin).flashing("error" -> Messages("admin.badRequest"))),

      tagFormData => {
        var language = "english"

        if (request.cookies.get("PLAY_LANG") == None) {}
        else request.cookies.get("PLAY_LANG").get.value match {
          case "fr" => language = "french"
          case _ => {}
        }

        val thisTag = Await.result(Tag.find(UUID(id)), 5 seconds).get

        val newTag = Await.result(Tag.findByName(tagFormData.name, language), 5 seconds)
        var count = 0
        var contentTitles = ""

        newTag match {
          case Some(tag) => {
            // Tag already exists. Untagging contents with old tag to the new tag.
            Await.result(ContentModel.findByTags(thisTag.name, language).map { contents =>
              count = contents.length
              contents.map { content =>

                if (contentTitles == "") contentTitles += content.title
                else contentTitles += ", " + content.title

                val futureTags = Tag.findByTagged(content.id.bytes)
                val foundTags = Await.result(futureTags, 5 seconds)
                var tagBuf = foundTags.map(_.name).toBuffer
                tagBuf.remove(tagBuf.indexOf(thisTag.name))
                Tag.untagContent(content.id.bytes)

                tagBuf += tag.name
                Tag.tagContent(tagBuf.mkString(","), content.id.bytes, language)

                if (content.primary_tag == thisTag.name) {
                  ContentModel.setPrimaryTag(content.id.bytes, tag.name)
                }
              }
            }, 5 seconds)
            Tag.delete(UUID(id))
            Future(Redirect("/admin#/manage/tags").flashing("success" -> ("This tag has been successfully merged with " + tag.name + " and the following " + count + " content entries have been updated: " + contentTitles)))

          }
          case None => {
            val futureTag = Tag.update(new Tag( id = UUID(id),
                                                name = tagFormData.name))

            for {
              maybeTag <- futureTag
            }
            yield {
              Redirect("/admin#/manage/tags").flashing("success" -> ("This tag has been successfully edited."))
            }
          }
        }
      }
    )
  }

  def deleteTag(id: String) = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    Tag.delete(UUID(id)).map { result =>
      result match {
        case true => Redirect("/admin#/manage/tags").flashing("success" -> (Messages("index.delete-tag")))
        case false => Redirect("/admin#/manage/tags").flashing("error" -> (Messages("index.badRequest")))
      }
    }
  }

}
