package controllers.api

import play.api.mvc.Controller
import play.api.cache.Cache
import play.api.data.Form
import play.api.data.Forms._
import play.api.i18n.Messages
import play.api.libs.json.{JsObject, Json}
import play.api.libs.concurrent.Execution.Implicits._

import scala.concurrent.Future
import scala.Some

import models.{Text, TextFormData, Tag}
import helpers.{UUID, Util, JsMessages}

object Texts extends Controller with controllers.Secured {

  def textForm = Form(
    mapping(
      "english-title" -> text,
      "french-title" -> text,
      "english-slug" -> text,
      "french-slug" -> text,
      "english-description" -> text,
      "french-description" -> text,
      "english-content" -> text,
      "french-content" -> text,
      "english-primary-tag" -> text,
      "french-primary-tag" -> text,
      "english-tags" -> text,
      "french-tags" -> text,
      "starred" -> number
    )(TextFormData.apply)(TextFormData.unapply)
  )

  def addText = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    textForm.bindFromRequest.fold(
      formWithErrors => Future.successful(Ok(Messages("admin.badRequest"))),

      textFormData => {
        val englishId = UUID.random
        val frenchId = UUID.random
        val englishPrimaryTag = textFormData.english_tags.split(",").distinct(0).trim
        val frenchPrimaryTag = textFormData.french_tags.split(",").distinct(0).trim
        val englishSlug = {
          if (textFormData.english_slug == "") {
            Util.slugify(englishPrimaryTag, textFormData.english_title)
          }
          else {
            Util.slugify(englishPrimaryTag, textFormData.english_slug)
          }
        }
        val frenchSlug = {
          if (textFormData.french_slug == "") {
            Util.slugify(frenchPrimaryTag, textFormData.french_title)
          }
          else {
            Util.slugify(frenchPrimaryTag, textFormData.french_slug)
          }
        }

        val futureEnglishText = Text.create(new Text( id = englishId,
                                                      translated_id = frenchId,
                                                      title = textFormData.english_title,
                                                      slug = englishSlug,
                                                      description = textFormData.english_description,
                                                      primary_tag = englishPrimaryTag,
                                                      starred = textFormData.starred,
                                                      content = textFormData.english_content), "english", user)

        val futureFrenchText = Text.create(new Text(  id = frenchId,
                                                      translated_id = englishId,
                                                      title = textFormData.french_title,
                                                      slug = frenchSlug,
                                                      description = textFormData.french_description,
                                                      primary_tag = frenchPrimaryTag,
                                                      starred = textFormData.starred,
                                                      content = textFormData.french_content), "french", user)

        for {
          maybeEnglishText <- futureEnglishText
          maybeFrenchText <- futureFrenchText
        }
        yield {
          (maybeEnglishText, maybeFrenchText) match {
            case (Some(english), Some(french)) => {
              Tag.tagContent(textFormData.english_tags, english.id.bytes, "english")
              Tag.tagContent(textFormData.french_tags, french.id.bytes, "french")
            }
            case _ => {}
          }
          Redirect(controllers.routes.Tagger.admin).flashing("success" -> Messages("admin.text-added"))
        }
      }
    )
  }

  def editText(english_id: String, french_id: String) = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    textForm.bindFromRequest.fold(
      formWithErrors => Future.successful(Redirect(controllers.routes.Tagger.admin).flashing("error" -> Messages("admin.badRequest"))),

      textFormData => {
        val englishSlug = {
          if (textFormData.english_slug == "") {
            Util.slugify(textFormData.english_primary_tag, textFormData.english_title)
          }
          else {
            textFormData.english_slug
          }
        }
        val frenchSlug = {
          if (textFormData.french_slug == "") {
            Util.slugify(textFormData.french_primary_tag, textFormData.french_title)
          }
          else {
            textFormData.french_slug
          }
        }

        val futureEnglishText = Text.update(new Text( id = UUID(english_id),
                                                      translated_id = UUID.random,
                                                      title = textFormData.english_title,
                                                      slug = englishSlug,
                                                      description = textFormData.english_description,
                                                      primary_tag = textFormData.english_primary_tag,
                                                      starred = textFormData.starred,
                                                      content = textFormData.english_content), user)

        val futureFrenchText = Text.update(new Text(id = UUID(french_id),
                                                    translated_id = UUID.random,
                                                    title = textFormData.french_title,
                                                    slug = frenchSlug,
                                                    description = textFormData.french_description,
                                                    primary_tag = textFormData.french_primary_tag,
                                                    starred = textFormData.starred,
                                                    content = textFormData.french_content), user)

        for {
          maybeEnglishText <- futureEnglishText
          maybeFrenchText <- futureFrenchText
        }
        yield {
          (maybeEnglishText, maybeFrenchText) match {
            case (Some(english), Some(french)) => {
              Tag.tagContent(textFormData.english_tags, UUID(english_id).bytes, "english")
              Tag.tagContent(textFormData.french_tags, UUID(french_id).bytes, "french")
            }
            case _ => {}
          }
          Redirect("/admin#/edit/tagged/content").flashing("success" -> (Messages("admin.text-edited")))
        }
      }
    )
  }

}
