package controllers.api

import play.api.mvc.Controller
import play.api.cache.Cache
import play.api.data.Form
import play.api.data.Forms._
import play.api.i18n.Messages
import play.api.libs.json.{JsObject, Json}
import play.api.libs.concurrent.Execution.Implicits._

import scala.concurrent.Future
import scala.Some

import models.{Video, VideoFormData, Tag}
import helpers.{UUID, Util, JsMessages}

object Videos extends Controller with controllers.Secured {

  def videoForm = Form(
    mapping(
      "english-title" -> text,
      "french-title" -> text,
      "english-slug" -> text,
      "french-slug" -> text,
      "english-description" -> text,
      "french-description" -> text,
      "english-long-description" -> text,
      "french-long-description" -> text,
      "video_type" -> text,
      "english_video_url" -> text,
      "french_video_url" -> text,
      "english-primary-tag" -> text,
      "french-primary-tag" -> text,
      "english-tags" -> text,
      "french-tags" -> text,
      "starred" -> number
    )(VideoFormData.apply)(VideoFormData.unapply)
  )

  def addVideo = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    videoForm.bindFromRequest.fold(
      formWithErrors => Future.successful(Ok(Messages("admin.badRequest"))),

      videoFormData => {
        val englishId = UUID.random
        val frenchId = UUID.random
        val englishPrimaryTag = videoFormData.english_tags.split(",").distinct(0).trim
        val frenchPrimaryTag = videoFormData.french_tags.split(",").distinct(0).trim
        val englishSlug = {
          if (videoFormData.english_slug == "") {
            Util.slugify(englishPrimaryTag, videoFormData.english_title)
          }
          else {
            Util.slugify(englishPrimaryTag, videoFormData.english_slug)
          }
        }
        val frenchSlug = {
          if (videoFormData.french_slug == "") {
            Util.slugify(frenchPrimaryTag, videoFormData.french_title)
          }
          else {
            Util.slugify(frenchPrimaryTag, videoFormData.french_slug)
          }
        }

        val futureEnglishVideo = Video.create(new Video(  id = englishId,
                                                          translated_id = frenchId,
                                                          title = videoFormData.english_title,
                                                          slug = englishSlug,
                                                          description = videoFormData.english_description,
                                                          long_description = videoFormData.english_long_description,
                                                          primary_tag = englishPrimaryTag,
                                                          starred = videoFormData.starred,
                                                          video_type = videoFormData.video_type,
                                                          video_url = videoFormData.english_video_url), "english", user)

        val futureFrenchVideo = Video.create(new Video( id = frenchId,
                                                        translated_id = englishId,
                                                        title = videoFormData.french_title,
                                                        slug = frenchSlug,
                                                        description = videoFormData.french_description,
                                                        long_description = videoFormData.french_long_description,
                                                        primary_tag = frenchPrimaryTag,
                                                        starred = videoFormData.starred,
                                                        video_type = videoFormData.video_type,
                                                        video_url = videoFormData.french_video_url), "french", user)

        for {
          maybeEnglishVideo <- futureEnglishVideo
          maybeFrenchVideo <- futureFrenchVideo
        }
        yield {
          (maybeEnglishVideo, maybeFrenchVideo) match {
            case (Some(english), Some(french)) => {
              Tag.tagContent(videoFormData.english_tags, english.id.bytes, "english")
              Tag.tagContent(videoFormData.french_tags, french.id.bytes, "french")
            }
            case _ => {}
          }
          Redirect(controllers.routes.Tagger.admin).flashing("success" -> Messages("admin.video-added"))
        }
      }
    )
  }

  def editVideo(english_id: String, french_id: String) = IsAuthenticatedAsync(parse.anyContent) { implicit user => implicit request =>
    videoForm.bindFromRequest.fold(
      formWithErrors => Future.successful(Redirect(controllers.routes.Tagger.admin).flashing("error" -> Messages("admin.badRequest"))),

      videoFormData => {
        val englishSlug = {
          if (videoFormData.english_slug == "") {
            Util.slugify(videoFormData.english_primary_tag, videoFormData.english_title)
          }
          else {
            videoFormData.english_slug
          }
        }
        val frenchSlug = {
          if (videoFormData.french_slug == "") {
            Util.slugify(videoFormData.french_primary_tag, videoFormData.french_title)
          }
          else {
            videoFormData.french_slug
          }
        }

        val futureEnglishVideo = Video.update(new Video( id = UUID(english_id),
                                                      translated_id = UUID.random,
                                                      title = videoFormData.english_title,
                                                      slug = englishSlug,
                                                      description = videoFormData.english_description,
                                                      long_description = videoFormData.english_long_description,
                                                      primary_tag = videoFormData.english_primary_tag,
                                                      starred = videoFormData.starred,
                                                      video_type = videoFormData.video_type,
                                                      video_url = videoFormData.english_video_url), user)

        val futureFrenchVideo = Video.update(new Video(id = UUID(french_id),
                                                    translated_id = UUID.random,
                                                    title = videoFormData.french_title,
                                                    slug = frenchSlug,
                                                    description = videoFormData.french_description,
                                                    long_description = videoFormData.french_long_description,
                                                    primary_tag = videoFormData.french_primary_tag,
                                                    starred = videoFormData.starred,
                                                    video_type = videoFormData.video_type,
                                                    video_url = videoFormData.french_video_url), user)

        for {
          maybeEnglishVideo <- futureEnglishVideo
          maybeFrenchVideo <- futureFrenchVideo
        }
        yield {
          (maybeEnglishVideo, maybeFrenchVideo) match {
            case (Some(english), Some(french)) => {
              Tag.tagContent(videoFormData.english_tags, UUID(english_id).bytes, "english")
              Tag.tagContent(videoFormData.french_tags, UUID(french_id).bytes, "french")
            }
            case _ => {}
          }
          Redirect("/admin#/edit/tagged/content").flashing("success" -> (Messages("admin.video-edited")))
        }
      }
    )
  }

}
