package models

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports._

import play.api.cache.Cache
import play.api.libs.json.Writes
import play.api.libs.functional.syntax._
import play.api.Play
import play.api.Play.current

import scala.concurrent.Future

import services.datasource.AppTDG

import helpers.UUID

case class App(
	id: UUID,
	translated_id: UUID,
	version: Long = 0,
	title: String,
	slug: String,
	description: String,
	content_type: String = "app",
	primary_tag: String,
	starred: Integer = 9,
	content: String,
	status: Integer = 1,
	createdAt: Option[DateTime] = None,
	updatedAt: Option[DateTime] = None
) extends Content {

	override def duplicate(): Content = {
  	this.copy()
  }
}

case class AppFormData(
  var english_title: String,
  var french_title: String,
  var english_slug: String,
  var french_slug: String,
  var english_description: String,
  var french_description: String,
  var english_content: String,
  var french_content: String,
  var english_tags: String,
  var french_tags: String,
  var starred: Int)

object App {

	/**
	 * Find all apps.
	 *
	 */
  def findAll: Future[Seq[App]] = {
    AppTDG.list.map { resultSet =>
      resultSet.map {
        item => rowToApp(item)
      }
    }
  }

	/**
	 * @param app the app to create in the database
	 * @return an optional app depending on whether the operation was successful
	 */
	def create(app: App, language: String, user: User): Future[Option[App]] = {
		val urlchars = "abcdefghijklmnopqrstuvwxyz0123456789-_.~/"
		val buf = new StringBuilder
		val s = app.slug.toLowerCase
		for (c <- s) if (urlchars contains c) buf.append( c ) else if (c == ' ') buf.append( '-' )

		AppTDG.insert(
			app.id.bytes,
			app.translated_id.bytes,
			app.title,
			buf.toString,
			app.description,
			app.content_type,
			app.primary_tag,
			app.starred,
			app.content,
			language,
			user.id.bytes
		).map {
			case Some(newVersion) => {
				val updatedApp = app.copy(version = newVersion)
				Cache.set("app." + updatedApp.id, updatedApp)
				Some(updatedApp)
			}
			case _ => None
		}
	}

  def update(app: App, user: User): Future[Option[App]] = {
		val urlchars = "abcdefghijklmnopqrstuvwxyz0123456789-_.~/"
		val buf = new StringBuilder
		val s = app.slug.toLowerCase
		for (c <- s) if (urlchars contains c) buf.append( c ) else if (c == ' ') buf.append( '-' )

    AppTDG.update(
			app.id.bytes,
			app.title,
			buf.toString,
			app.description,
			app.content,
			app.primary_tag,
			app.starred,
			user.id.bytes
    ).map {
			case Some(newVersion) => {
				val updatedApp = app.copy(version = newVersion)
				Cache.set("app." + updatedApp.id, updatedApp)
				Some(updatedApp)
			}
      case _ => None
    }
  }

	/**
	 * Converts a RowData object into a App.
	 *
	 * @param row the row data to convert
	 * @return a app object
	 */
	def rowToApp(row: RowData): App = {
		App(
			id           					= UUID(row("id").asInstanceOf[Array[Byte]]),
			translated_id 				= UUID(row("translated_id").asInstanceOf[Array[Byte]]),
			version     				  = row("version").asInstanceOf[Long],
			title 								= row("title").asInstanceOf[String],
			slug 									= row("slug").asInstanceOf[String],
			description 				  = row("description").asInstanceOf[String],
			content_type 				  = row("content_type").asInstanceOf[String],
			primary_tag 					= row("primary_tag").asInstanceOf[String],
			starred 							= row("starred").asInstanceOf[Integer],
			content 							= row("content").asInstanceOf[String],
			status 								= row("status").asInstanceOf[Integer],
			createdAt   					= Some(row("created_at").asInstanceOf[DateTime]),
			updatedAt   					= Some(row("updated_at").asInstanceOf[DateTime])
		)
	}

}
