package models

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports._

import play.api.cache.Cache
import play.api.libs.json.{Writes, JsValue, Json}
import play.api.libs.functional.syntax._
import play.api.Play.current

import scala.concurrent.Future

import services.datasource.AudioTDG

import helpers.UUID

case class Audio(
  id: UUID,
  translated_id: UUID,
  version: Long = 0,
  title: String,
  slug: String,
  description: String,
  content_type: String = "audio",
  primary_tag: String,
  starred: Integer = 9,
  audio_url: String,
  status: Integer = 1,
  createdAt: Option[DateTime] = None,
  updatedAt: Option[DateTime] = None
) extends Content {

  override def duplicate(): Content = {
    this.copy()
  }
}

case class AudioFormData(
  var english_title: String,
  var french_title: String,
  var english_slug: String,
  var french_slug: String,
  var english_description: String,
  var french_description: String,
  var english_audio_url: String,
  var french_audio_url: String,
  var english_primary_tag: String,
  var french_primary_tag: String,
  var english_tags: String,
  var french_tags: String,
  var starred: Int)

object Audio {

  implicit val audioWrites = new Writes[Audio] {
    def writes(audio: Audio): JsValue = {
      Json.obj(
        "id" -> audio.id.string,
        "translated_id" -> audio.translated_id.string,
        "title" -> audio.title,
        "slug" -> audio.slug,
        "description" -> audio.description,
        "content_type" -> audio.content_type,
        "primary_tag" -> audio.primary_tag,
        "starred" -> audio.starred.asInstanceOf[Int],
        "audio_url" -> audio.audio_url,
        "created_at" -> audio.createdAt
      )
    }
  }

  /**
   * Find all audios.
   */
  def findAll: Future[Seq[Audio]] = {
    AudioTDG.list.map { resultSet =>
      resultSet.map {
        item => rowToAudio(item)
      }
    }
  }

  /**
   * @param audio the audio to create in the database
   * @return an optional audio depending on whether the operation was successful
   */
  def create(audio: Audio, language: String, user: User): Future[Option[Audio]] = {
    val urlchars = "abcdefghijklmnopqrstuvwxyz0123456789-_.~/"
    val buf = new StringBuilder
    val s = audio.slug.toLowerCase
    for (c <- s) if (urlchars contains c) buf.append( c ) else if (c == ' ') buf.append( '-' )

    AudioTDG.insert(
      audio.id.bytes,
      audio.translated_id.bytes,
      audio.title,
      buf.toString,
      audio.description,
      audio.content_type,
      audio.primary_tag,
      audio.starred,
      audio.audio_url,
      language,
      user.id.bytes
    ).map {
      case Some(newVersion) => {
        val updatedAudio = audio.copy(version = newVersion)
        Cache.set("audio." + updatedAudio.id, updatedAudio)
        Some(updatedAudio)
      }
      case _ => None
    }
  }

  def update(audio: Audio, user: User): Future[Option[Audio]] = {
    val urlchars = "abcdefghijklmnopqrstuvwxyz0123456789-_.~/"
    val buf = new StringBuilder
    val s = audio.slug.toLowerCase
    for (c <- s) if (urlchars contains c) buf.append( c ) else if (c == ' ') buf.append( '-' )

    AudioTDG.update(
      audio.id.bytes,
      audio.title,
      buf.toString,
      audio.description,
      audio.audio_url,
      audio.primary_tag,
      audio.starred,
      user.id.bytes
    ).map {
      case Some(newVersion) => {
        val updatedAudio = audio.copy(version = newVersion)
        Cache.set("audio." + updatedAudio.id, updatedAudio)
        Some(updatedAudio)
      }
      case _ => None
    }
  }

  /**
   * Converts a RowData object into a Audio.
   *
   * @param row the row data to convert
   * @return a audio object
   */
  def rowToAudio(row: RowData): Audio = {
    Audio(
      id                    = UUID(row("id").asInstanceOf[Array[Byte]]),
      translated_id         = UUID(row("translated_id").asInstanceOf[Array[Byte]]),
      version               = row("version").asInstanceOf[Long],
      title                 = row("title").asInstanceOf[String],
      slug                  = row("slug").asInstanceOf[String],
      description           = row("description").asInstanceOf[String],
      content_type          = row("content_type").asInstanceOf[String],
      primary_tag           = row("primary_tag").asInstanceOf[String],
      starred               = row("starred").asInstanceOf[Integer],
      audio_url             = row("audio_url").asInstanceOf[String],
      status                = row("status").asInstanceOf[Integer],
      createdAt             = Some(row("created_at").asInstanceOf[DateTime]),
      updatedAt             = Some(row("updated_at").asInstanceOf[DateTime])
    )
  }

}
