package models

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports._

import play.api.cache.Cache
import play.api.libs.json.{Writes, JsValue, Json}
import play.api.libs.functional.syntax._
import play.api.Play.current

import scala.concurrent.Future

import services.datasource.ContentTDG

import helpers.UUID

object ContentModel {

  /**
   * Find single contentModel.
   *
   * @param id the UUID of the contentModel
   * @return an optional contentModel
   */
  def find(id: UUID): Future[Option[Content]] = {
    ContentTDG.find(id.bytes).map { result =>
      result match {
        case IndexedSeq(item) => Some(rowToContent(item))
        case _ => None
      }
    }
  }

  def findByTitle(title: String): Future[Option[Content]] = {
    ContentTDG.findByTitle(title).map { result =>
      result match {
        case IndexedSeq(item) => Some(rowToContent(item))
        case _ => None
      }
    }
  }

  def findBySlug(slug: String): Future[Option[Content]] = {
    ContentTDG.findBySlug(slug).map { result =>
      result match {
        case IndexedSeq(item) => Some(rowToContent(item))
        case _ => None
      }
    }
  }

  def findSlug(id: UUID): Future[Option[String]] = {
    ContentTDG.findSlug(id.bytes).map { result =>
      result match {
        case IndexedSeq(item) => Some(item("slug").asInstanceOf[String])
        case _ => None
      }
    }
  }

  def findLanguage(id: UUID): Future[Option[String]] = {
    ContentTDG.findLanguage(id.bytes).map { result =>
      result match {
        case IndexedSeq(item) => Some(item("language").asInstanceOf[String])
        case _ => None
      }
    }
  }

  def findByLanguage(language: String): Future[Seq[Content]] = {
    ContentTDG.findByLanguage(language).map { resultSet =>
      resultSet.map {
        item => rowToContent(item)
      }
    }
  }

  def findUntagged(language: String): Future[Seq[Content]] = {
    ContentTDG.findUntagged(language).map { resultSet =>
      resultSet.map {
        item => rowToContent(item)
      }
    }
  }

  /**
   * Find all contentModels.
   */
  def findAll: Future[Seq[Content]] = {
    ContentTDG.list.map { resultSet =>
      resultSet.map {
        item => rowToContent(item)
      }
    }
  }

  def findByTags(tags: String, language: String): Future[Seq[Content]] = {
    ContentTDG.findByTags(tags, language).map { resultSet =>
      resultSet.map {
        item => rowToContent(item)
      }
    }
  }

  def setPrimaryTag(contentId: Array[Byte], primaryTag: String): Future[Boolean] = {
    ContentTDG.setPrimaryTag(contentId, primaryTag)
  }

  def delete(id: UUID): Future[Boolean] = {
    ContentTDG.delete(id.bytes)
  }

  /**
   * Converts a RowData object into a ContentModel.
   *
   * @param row the row data to convert
   * @return a contentModel object
   */
  private def rowToContent(row: RowData): Content = {
    row("content_type").asInstanceOf[String] match {
      case "text" => Text.rowToText(row)

      case "video" => Video.rowToVideo(row)

      case "link" => Link.rowToLink(row)

      case "audio" => Audio.rowToAudio(row)

      case "tag_line" => TagLine.rowToTagLine(row)

      case "header_item" => HeaderItem.rowToHeaderItem(row)

      case "footer_item" => FooterItem.rowToFooterItem(row)

      case "site" => Site.rowToSite(row)

      case "slide" => Slide.rowToSlide(row)

      case "featured" => Featured.rowToFeatured(row)

      case "select_featured" => SelectFeatured.rowToSelectFeatured(row)

      case "site_word" => SiteWord.rowToSiteWord(row)

      case _ => Link.rowToLink(row)
    }
  }
}
