package models

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports.DateTime

import play.api.cache.Cache
import play.api.libs.json.{Writes, JsValue, Json}
import play.api.libs.functional.syntax._
import play.api.Play.current

import scala.concurrent.Future

import services.datasource.FeaturedTDG

import helpers.UUID

case class Featured(
	id: UUID,
	translated_id: UUID,
	version: Long = 0,
	title: String,
	slug: String,
	description: String,
	position: Integer,
	icon_type: String,
	content: String,
	content_type: String = "featured",
	primary_tag: String,
	starred: Integer = 9,
	status: Integer = 1,
	createdAt: Option[DateTime] = None,
	updatedAt: Option[DateTime] = None
) extends Content {

	override def duplicate(): Content = {
  	this.copy()
  }
}

case class FeaturedFormData(
  var english_title: String,
  var french_title: String,
  var english_slug: String,
  var french_slug: String,
  var english_description: String,
  var french_description: String,
  var position: Int,
  var icon_type: String,
  var english_content: String,
  var french_content: String,
  var english_primary_tag: String,
  var french_primary_tag: String,
  var starred: Int)

object Featured {

  implicit val featuredWrites = new Writes[Featured] {
    def writes(featured: Featured): JsValue = {
      Json.obj(
        "id" -> featured.id.string,
        "translated_id" -> featured.translated_id.string,
        "title" -> featured.title,
        "slug" -> featured.slug,
        "description" -> featured.description,
        "featured_position" -> featured.position.asInstanceOf[Int],
        "icon_type" -> featured.icon_type,
        "featured_content" -> featured.content,
        "content_type" -> featured.content_type,
        "primary_tag" -> featured.primary_tag,
        "starred" -> featured.starred.asInstanceOf[Int],
        "created_at" -> featured.createdAt
      )
    }
  }

	/**
	 * Find all featureds.
	 *
	 */
  def findAll: Future[Seq[Featured]] = {
    FeaturedTDG.list.map { resultSet =>
      resultSet.map {
        item => rowToFeatured(item)
      }
    }
  }

  def findByLanguage(language: String): Future[Seq[Featured]] = {
    FeaturedTDG.findByLanguage(language).map { resultSet =>
      resultSet.map {
        item => rowToFeatured(item)
      }
    }
  }

	/**
	 * @param featured the featured to create in the database
	 * @return an optional featured depending on whether the operation was successful
	 */
	def create(featured: Featured, language: String, user: User): Future[Option[Featured]] = {
		val urlchars = "abcdefghijklmnopqrstuvwxyz0123456789-_.~/"
		val buf = new StringBuilder
		val s = featured.slug.toLowerCase
		for (c <- s) if (urlchars contains c) buf.append( c ) else if (c == ' ') buf.append( '-' )

		FeaturedTDG.insert(
			featured.id.bytes,
			featured.translated_id.bytes,
			featured.title,
			buf.toString,
			featured.description,
			featured.position,
			featured.icon_type,
			featured.content,
			featured.content_type,
			featured.primary_tag,
			featured.starred,
			language,
			user.id.bytes
		).map {
			case Some(newVersion) => {
				val updatedFeatured = featured.copy(version = newVersion)
				Cache.set("featured." + updatedFeatured.id, updatedFeatured)
				Some(updatedFeatured)
			}
			case _ => None
		}
	}

  def update(featured: Featured, user: User): Future[Option[Featured]] = {
		val urlchars = "abcdefghijklmnopqrstuvwxyz0123456789-_.~/"
		val buf = new StringBuilder
		val s = featured.slug.toLowerCase
		for (c <- s) if (urlchars contains c) buf.append( c ) else if (c == ' ') buf.append( '-' )

    FeaturedTDG.update(
			featured.id.bytes,
			featured.title,
			buf.toString,
			featured.description,
			featured.position,
			featured.icon_type,
			featured.content,
			featured.primary_tag,
			featured.starred,
			user.id.bytes
    ).map {
			case Some(newVersion) => {
				val updatedFeatured = featured.copy(version = newVersion)
				Cache.set("featured" + updatedFeatured.id, updatedFeatured)
				Some(updatedFeatured)
			}
      case _ => None
    }
  }

	/**
	 * Converts a RowData object into a Featured.
	 *
	 * @param row the row data to convert
	 * @return a featured object
	 */
	def rowToFeatured(row: RowData): Featured = {
		Featured(
			id           					= UUID(row("id").asInstanceOf[Array[Byte]]),
			translated_id 				= UUID(row("translated_id").asInstanceOf[Array[Byte]]),
			version     				  = row("version").asInstanceOf[Long],
			title 								= row("title").asInstanceOf[String],
			slug 									= row("slug").asInstanceOf[String],
			description 				  = row("description").asInstanceOf[String],
			position 							= row("featured_position").asInstanceOf[Integer],
			icon_type 						= row("icon_type").asInstanceOf[String],
			content 							= row("featured_content").asInstanceOf[String],
			content_type 				  = row("content_type").asInstanceOf[String],
			primary_tag 					= row("primary_tag").asInstanceOf[String],
			starred 							= row("starred").asInstanceOf[Integer],
			status 								= row("status").asInstanceOf[Integer],
			createdAt   					= Some(row("created_at").asInstanceOf[DateTime]),
			updatedAt   					= Some(row("updated_at").asInstanceOf[DateTime])
		)
	}

}
