package models

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports.DateTime

import play.api.cache.Cache
import play.api.libs.json.{Writes, JsValue, Json}
import play.api.libs.functional.syntax._
import play.api.Play.current

import scala.concurrent.Future

import services.datasource.FooterItemTDG

import helpers.UUID

case class FooterItem(
	id: UUID,
	translated_id: UUID,
	version: Long = 0,
	title: String,
	slug: String,
	description: String,
	position: Integer,
	content: String,
	content_type: String = "footer_item",
	primary_tag: String = "footer_item",
	starred: Integer = 9,
	status: Integer = 1,
	createdAt: Option[DateTime] = None,
	updatedAt: Option[DateTime] = None
) extends Content {

	override def duplicate(): Content = {
  	this.copy()
  }
}

case class FooterItemFormData(
  var english_title: String,
  var french_title: String,
  var english_description: String,
  var french_description: String,
  var position: Int,
  var english_content: String,
  var french_content: String)

object FooterItem {

	/**
	 * Find single footerItem.
	 *
	 * @param id the UUID of the footerItem
	 * @return an optional footerItem
	 */
	def find(id: UUID): Future[Option[FooterItem]] = {
    FooterItemTDG.find(id.bytes).map { result =>
    	result match {
    		case IndexedSeq(item) => Some(rowToFooterItem(item))
    		case _ => None
    	}
    }
	}

	/**
	 * Find all footerItems.
	 *
	 */
  def findAll: Future[Seq[FooterItem]] = {
    FooterItemTDG.list.map { resultSet =>
      resultSet.map {
        item => rowToFooterItem(item)
      }
    }
  }

  def findByLanguage(language: String): Future[Seq[FooterItem]] = {
    FooterItemTDG.findByLanguage(language).map { resultSet =>
      resultSet.map {
        item => rowToFooterItem(item)
      }
    }
  }

	/**
	 * @param footerItem the footerItem to create in the database
	 * @return an optional footerItem depending on whether the operation was successful
	 */
	def create(footerItem: FooterItem, language: String, user: User): Future[Option[FooterItem]] = {
		FooterItemTDG.insert(
			footerItem.id.bytes,
			footerItem.translated_id.bytes,
			footerItem.title,
			footerItem.slug,
			footerItem.description,
			footerItem.position,
			footerItem.content,
			footerItem.content_type,
			language,
			user.id.bytes
		).map {
			case Some(newVersion) => {
				val updatedFooterItem = footerItem.copy(version = newVersion)
				Cache.set("footerItem." + updatedFooterItem.id, updatedFooterItem)
				Some(updatedFooterItem)
			}
			case _ => None
		}
	}

  def update(footerItem: FooterItem, user: User): Future[Option[FooterItem]] = {
    FooterItemTDG.update(
			footerItem.id.bytes,
			footerItem.title,
			footerItem.slug,
			footerItem.description,
			footerItem.position,
			footerItem.content,
			user.id.bytes
    ).map {
			case Some(newVersion) => {
				val updatedFooterItem = footerItem.copy(version = newVersion)
				Cache.set("footerItem" + updatedFooterItem.id, updatedFooterItem)
				Some(updatedFooterItem)
			}
      case _ => None
    }
  }

	/**
	 * Converts a RowData object into a FooterItem.
	 *
	 * @param row the row data to convert
	 * @return a footerItem object
	 */
	def rowToFooterItem(row: RowData): FooterItem = {
		FooterItem(
			id           					= UUID(row("id").asInstanceOf[Array[Byte]]),
			translated_id 				= UUID(row("translated_id").asInstanceOf[Array[Byte]]),
			version     				  = row("version").asInstanceOf[Long],
			title 								= row("title").asInstanceOf[String],
			slug 									= UUID.random.string,
			description 				  = row("description").asInstanceOf[String],
			position 							= row("footer_item_position").asInstanceOf[Integer],
			content 							= row("footer_content").asInstanceOf[String],
			content_type 				  = row("content_type").asInstanceOf[String],
			status 								= row("status").asInstanceOf[Integer],
			createdAt   					= Some(row("created_at").asInstanceOf[DateTime]),
			updatedAt   					= Some(row("updated_at").asInstanceOf[DateTime])
		)
	}

}
