package models

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports.DateTime

import play.api.cache.Cache
import play.api.libs.json.{Writes, JsValue, Json}
import play.api.libs.functional.syntax._
import play.api.Play.current

import scala.concurrent.Future

import services.datasource.HeaderItemTDG

import helpers.UUID

case class HeaderItem(
	id: UUID,
	translated_id: UUID,
	version: Long = 0,
	title: String,
	slug: String,
	description: String,
	position: Integer,
	content: String,
	content_type: String = "header_item",
	primary_tag: String = "header_item",
	starred: Integer = 9,
	status: Integer = 1,
	createdAt: Option[DateTime] = None,
	updatedAt: Option[DateTime] = None
) extends Content {

	override def duplicate(): Content = {
  	this.copy()
  }
}

case class HeaderItemFormData(
  var english_title: String,
  var french_title: String,
  var english_description: String,
  var french_description: String,
  var position: Int,
  var english_content: String,
  var french_content: String)

object HeaderItem {

  implicit val headerItemWrites = new Writes[HeaderItem] {
    def writes(headerItem: HeaderItem): JsValue = {
      Json.obj(
        "id" -> headerItem.id.string,
        "translated_id" -> headerItem.translated_id.string,
        "title" -> headerItem.title,
        "description" -> headerItem.description,
        "header_item_position" -> headerItem.position.asInstanceOf[Int],
        "header_content" -> headerItem.content,
        "content_type" -> headerItem.content_type,
        "created_at" -> headerItem.createdAt
      )
    }
  }

	/**
	 * Find single headerItem.
	 *
	 * @param id the UUID of the headerItem
	 * @return an optional headerItem
	 */
	def find(id: UUID): Future[Option[HeaderItem]] = {
    HeaderItemTDG.find(id.bytes).map { result =>
    	result match {
    		case IndexedSeq(item) => Some(rowToHeaderItem(item))
    		case _ => None
    	}
    }
	}

	/**
	 * Find all headerItems.
	 *
	 */
  def findAll: Future[Seq[HeaderItem]] = {
    HeaderItemTDG.list.map { resultSet =>
      resultSet.map {
        item => rowToHeaderItem(item)
      }
    }
  }

  def findByLanguage(language: String): Future[Seq[HeaderItem]] = {
    HeaderItemTDG.findByLanguage(language).map { resultSet =>
      resultSet.map {
        item => rowToHeaderItem(item)
      }
    }
  }

  def setPosition(id: String, position: Int): Future[Boolean] = {
    HeaderItemTDG.setPosition(UUID(id).bytes, position)
  }

	/**
	 * @param headerItem the headerItem to create in the database
	 * @return an optional headerItem depending on whether the operation was successful
	 */
	def create(headerItem: HeaderItem, language: String, user: User): Future[Option[HeaderItem]] = {
		HeaderItemTDG.insert(
			headerItem.id.bytes,
			headerItem.translated_id.bytes,
			headerItem.title,
			headerItem.slug,
			headerItem.description,
			headerItem.position,
			headerItem.content,
			headerItem.content_type,
			language,
			user.id.bytes
		).map {
			case Some(newVersion) => {
				val updatedHeaderItem = headerItem.copy(version = newVersion)
				Cache.set("headerItem." + updatedHeaderItem.id, updatedHeaderItem)
				Some(updatedHeaderItem)
			}
			case _ => None
		}
	}

  def update(headerItem: HeaderItem, user: User): Future[Option[HeaderItem]] = {
    HeaderItemTDG.update(
			headerItem.id.bytes,
			headerItem.title,
			headerItem.slug,
			headerItem.description,
			headerItem.position,
			headerItem.content,
			user.id.bytes
    ).map {
			case Some(newVersion) => {
				val updatedHeaderItem = headerItem.copy(version = newVersion)
				Cache.set("headerItem" + updatedHeaderItem.id, updatedHeaderItem)
				Some(updatedHeaderItem)
			}
      case _ => None
    }
  }

	/**
	 * Converts a RowData object into a HeaderItem.
	 *
	 * @param row the row data to convert
	 * @return a headerItem object
	 */
	def rowToHeaderItem(row: RowData): HeaderItem = {
		HeaderItem(
			id           					= UUID(row("id").asInstanceOf[Array[Byte]]),
			translated_id 				= UUID(row("translated_id").asInstanceOf[Array[Byte]]),
			version     				  = row("version").asInstanceOf[Long],
			title 								= row("title").asInstanceOf[String],
			slug 									= UUID.random.string,
			description 				  = row("description").asInstanceOf[String],
			position 							= row("header_item_position").asInstanceOf[Integer],
			content 							= row("header_content").asInstanceOf[String],
			content_type 				  = row("content_type").asInstanceOf[String],
			status 								= row("status").asInstanceOf[Integer],
			createdAt   					= Some(row("created_at").asInstanceOf[DateTime]),
			updatedAt   					= Some(row("updated_at").asInstanceOf[DateTime])
		)
	}

}
