package models

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports._

import play.api.cache.Cache
import play.api.Play.current
import play.api.libs.json._
import play.api.libs.functional.syntax._
import play.api.Play.current
import play.api.Logger

import scala.concurrent.Future

import helpers.UUID

import services.datasource.HomeBlockTDG

case class HomeBlock(
  id: String = UUID.random.string,
  version: Long = 0,
  language: String,
  title: String = "",
  description: String = "",
  content: String = ""
)

object HomeBlock {

  implicit val homeBlockReads = (
    (__ \ "id").read[String] and
    (__ \ "version").read[Long] and
    (__ \ "language").read[String] and
    (__ \ "title").read[String] and
    (__ \ "description").read[String] and
    (__ \ "content").read[String]
  )(HomeBlock.apply _)

  implicit val homeBlockWrites = new Writes[HomeBlock] {
    def writes(homeBlock: HomeBlock): JsValue = {
      Json.obj(
        "id" -> homeBlock.id,
        "version" -> homeBlock.version,
        "language" -> homeBlock.language,
        "title" -> homeBlock.title,
        "description" -> homeBlock.description,
        "content" -> homeBlock.content
      )
    }
  }

  /**
   * Find a HomeBlock by id.
   *
   * @param id the id of the HomeBlock
   * @return a future optional HomeBlock depending on whether the HomeBlock exists
   */
  def findById(id: String, language: String): Future[Option[HomeBlock]] = {
    HomeBlockTDG.findById(UUID(id).bytes, language).map { rows =>
      rows.map(rowToHomeBlock)
    }
  }

  def findByLanguage(language: String): Future[Option[HomeBlock]] = {
    HomeBlockTDG.findByLanguage(language).map { rows =>
      rows.map(rowToHomeBlock)
    }
  }

  /**
   * Inserts a HomeBlock into the database.
   *
   * @param homeBlock the HomeBlock to create in the database
   * @return a future optional HomeBlock depending on whether the operation was successful
   */
  def create(homeBlock: HomeBlock): Future[Option[HomeBlock]] = {
    HomeBlockTDG.insert(
      UUID(homeBlock.id).bytes,
      homeBlock.version,
      homeBlock.language,
      homeBlock.title,
      homeBlock.description,
      homeBlock.content
    ).map {
      case Some(newVersion) => {
        val updatedHomeBlock = homeBlock.copy(version = newVersion)
        Cache.set("homeBlock." + updatedHomeBlock.id, updatedHomeBlock)
        Some(updatedHomeBlock)
      }
      case None => None
    }
  }

  /**
   * Updates a HomeBlock in the database.
   *
   * @param homeBlock the HomeBlock to update in the database
   * @return a future optional HomeBlock depending on whether the operation was successful
   */
  def update(homeBlock: HomeBlock): Future[Option[HomeBlock]] = {
    HomeBlockTDG.update(
      UUID(homeBlock.id).bytes,
      homeBlock.version,
      homeBlock.language,
      homeBlock.title,
      homeBlock.description,
      homeBlock.content
    ).map {
      case Some(newVersion) => {
        val updatedHomeBlock = homeBlock.copy(version = newVersion)
        Cache.set("homeBlock." + updatedHomeBlock.id, updatedHomeBlock)
        Some(updatedHomeBlock)
      }
      case None => None
    }
  }

  /**
   * Deletes a HomeBlock from the database.
   *
   * @param id id of the HomeBlock to delete
   * @return a future boolean depending on whether the operation was successful
   */
  def delete(id: String): Future[Boolean] = {
    HomeBlockTDG.delete(UUID(id).bytes)
  }

  /**
   * Converts a RowData object into a HomeBlock.
   *
   * @param row the row data to convert
   * @return a HomeBlock object
   */
  private def rowToHomeBlock(row: RowData): HomeBlock = {
    HomeBlock(
      id          = UUID((row("id").asInstanceOf[Array[Byte]])).string,
      version     = row("version").asInstanceOf[Long],
      language    = row("language").asInstanceOf[String],
      title       = row("title").asInstanceOf[String],
      description = row("description").asInstanceOf[String],
      content     = row("content").asInstanceOf[String]
    )
  }

}
