package models

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports._

import play.api.cache.Cache
import play.api.libs.json.{Writes, JsValue, Json}
import play.api.libs.functional.syntax._
import play.api.Play.current

import scala.concurrent.Future

import services.datasource.LinkTDG

import helpers.UUID

case class Link(
  id: UUID,
  translated_id: UUID,
  version: Long = 0,
  title: String,
  slug: String,
  description: String,
  content_type: String = "link",
  primary_tag: String,
  starred: Integer = 9,
  link_url: String,
  status: Integer = 1,
  createdAt: Option[DateTime] = None,
  updatedAt: Option[DateTime] = None
) extends Content {

  override def duplicate(): Content = {
    this.copy()
  }
}

case class LinkFormData(
  var english_title: String,
  var french_title: String,
  var english_slug: String,
  var french_slug: String,
  var english_description: String,
  var french_description: String,
  var english_link_url: String,
  var french_link_url: String,
  var english_primary_tag: String,
  var french_primary_tag: String,
  var english_tags: String,
  var french_tags: String,
  var starred: Int)

object Link {

  implicit val linkWrites = new Writes[Link] {
    def writes(link: Link): JsValue = {
      Json.obj(
        "id" -> link.id.string,
        "translated_id" -> link.translated_id.string,
        "title"  -> link.title,
        "slug" -> link.slug,
        "description" -> link.description,
        "content_type" -> link.content_type,
        "primary_tag" -> link.primary_tag,
        "starred" -> link.starred.asInstanceOf[Int],
        "link_url" -> link.link_url,
        "created_at" -> link.createdAt
      )
    }
  }

  /**
   * Find all links.
   */
  def findAll: Future[Seq[Link]] = {
    LinkTDG.list.map { resultSet =>
      resultSet.map {
        item => rowToLink(item)
      }
    }
  }

  /**
   * @param link the link to create in the database
   * @return an optional link depending on whether the operation was successful
   */
  def create(link: Link, language: String, user: User): Future[Option[Link]] = {
    val urlchars = "abcdefghijklmnopqrstuvwxyz0123456789-_.~/"
    val buf = new StringBuilder
    val s = link.slug.toLowerCase
    for (c <- s) if (urlchars contains c) buf.append( c ) else if (c == ' ') buf.append( '-' )

    LinkTDG.insert(
      link.id.bytes,
      link.translated_id.bytes,
      link.title,
      buf.toString,
      link.description,
      link.content_type,
      link.primary_tag,
      link.starred,
      link.link_url,
      language,
      user.id.bytes
    ).map {
      case Some(newVersion) => {
        val updatedLink = link.copy(version = newVersion)
        Cache.set("link." + updatedLink.id, updatedLink)
        Some(updatedLink)
      }
      case _ => None
    }
  }

  def update(link: Link, user: User): Future[Option[Link]] = {
    val urlchars = "abcdefghijklmnopqrstuvwxyz0123456789-_.~/"
    val buf = new StringBuilder
    val s = link.slug.toLowerCase
    for (c <- s) if (urlchars contains c) buf.append( c ) else if (c == ' ') buf.append( '-' )

    LinkTDG.update(
      link.id.bytes,
      link.title,
      buf.toString,
      link.description,
      link.link_url,
      link.primary_tag,
      link.starred,
      user.id.bytes
    ).map {
      case Some(newVersion) => {
        val updatedLink = link.copy(version = newVersion)
        Cache.set("link." + updatedLink.id, updatedLink)
        Some(updatedLink)
      }
      case _ => None
    }
  }

  /**
   * Converts a RowData object into a Link.
   *
   * @param row the row data to convert
   * @return a link object
   */
  def rowToLink(row: RowData): Link = {
    Link(
      id                    = UUID(row("id").asInstanceOf[Array[Byte]]),
      translated_id         = UUID(row("translated_id").asInstanceOf[Array[Byte]]),
      version               = row("version").asInstanceOf[Long],
      title                 = row("title").asInstanceOf[String],
      slug                  = row("slug").asInstanceOf[String],
      description           = row("description").asInstanceOf[String],
      content_type          = row("content_type").asInstanceOf[String],
      primary_tag           = row("primary_tag").asInstanceOf[String],
      starred               = row("starred").asInstanceOf[Integer],
      link_url              = row("link_url").asInstanceOf[String],
      status                = row("status").asInstanceOf[Integer],
      createdAt             = Some(row("created_at").asInstanceOf[DateTime]),
      updatedAt             = Some(row("updated_at").asInstanceOf[DateTime])
    )
  }

}
