package models

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports._

import play.api.cache.Cache
import play.api.Play.current
import play.api.libs.json._
import play.api.libs.functional.syntax._
import play.api.Play.current
import play.api.Logger

import scala.concurrent.Future

import helpers.UUID

import services.datasource.MenuItemTDG

case class MenuItem(
  id: String = UUID.random.string,
  version: Long = 0,
  parentId: Option[String],
  tagLineId: Option[String],
  contentId: Option[String],
  language: String,
  position: Int = 0,
  title: String,
  description: String = "",
  longDescription: String = "",
  published: Int = 0
)

object MenuItem {

  implicit val menuItemReads = (
    (__ \ "id").read[String] and
    (__ \ "version").read[Long] and
    (__ \ "parentId").readNullable[String] and
    (__ \ "tagLineId").readNullable[String] and
    (__ \ "contentId").readNullable[String] and
    (__ \ "language").read[String] and
    (__ \ "position").read[Int] and
    (__ \ "title").read[String] and
    (__ \ "description").read[String] and
    (__ \ "longDescription").read[String] and
    (__ \ "published").read[Int]
  )(MenuItem.apply _)

  implicit val menuItemWrites = new Writes[MenuItem] {
    def writes(menuItem: MenuItem): JsValue = {
      Json.obj(
        "id" -> menuItem.id,
        "version" -> menuItem.version,
        "parentId" -> menuItem.parentId,
        "tagLineId" -> menuItem.tagLineId,
        "contentId" -> menuItem.contentId,
        "language" -> menuItem.language,
        "position" -> menuItem.position,
        "title" -> menuItem.title,
        "description" -> menuItem.description,
        "longDescription" -> menuItem.longDescription,
        "published" -> menuItem.published
      )
    }
  }

  /**
   * Find a MenuItem by id.
   *
   * @param id the id of the MenuItem
   * @return a future optional MenuItem depending on whether the MenuItem exists
   */
  def findById(id: String, language: String): Future[Option[MenuItem]] = {
    MenuItemTDG.findById(UUID(id).bytes, language).map { rows =>
      rows.map(rowToMenuItem)
    }
  }

  /**
   * Find all parent MenuItems.
   *
   * @return a future indexed sequence of MenuItems
   */
  def findAllParents(language: String): Future[IndexedSeq[MenuItem]] = {
    MenuItemTDG.findAllParents(language).map(_.map(rowToMenuItem))
  }

  /**
   * Find all child MenuItems that belong to a specific parent MenuItem.
   *
   * @param parentId the id of the parent MenuItem
   * @return a future indexed sequence of ManuItems
   */
  def findByParentId(parentId: String, language: String): Future[IndexedSeq[MenuItem]] = {
    MenuItemTDG.findByParentId(UUID(parentId).bytes, language).map(_.map(rowToMenuItem))
  }

  def getLargestPosition: Future[Option[Int]] = {
    MenuItemTDG.getLargestPosition
  }

  def getLargestPosition(parentId: String): Future[Option[Int]] = {
    MenuItemTDG.getLargestPosition(UUID(parentId).bytes)
  }

  def setPosition(id: String, position: Int): Future[Boolean] = {
    MenuItemTDG.setPosition(UUID(id).bytes, position)
  }

  /**
   * Inserts a MenuItem into the database.
   *
   * @param menuItem the MenuItem to create in the database
   * @return a future optional MenuItem depending on whether the operation was successful
   */
  def create(menuItem: MenuItem): Future[Option[MenuItem]] = {
    MenuItemTDG.insert(
      UUID(menuItem.id).bytes,
      menuItem.version,
      menuItem.parentId,
      menuItem.tagLineId,
      menuItem.contentId,
      menuItem.language,
      menuItem.position,
      menuItem.title,
      menuItem.description,
      menuItem.longDescription,
      menuItem.published
    ).map {
      case Some(newVersion) => {
        val updatedMenuItem = menuItem.copy(version = newVersion)
        Cache.set("menuItem." + updatedMenuItem.id, updatedMenuItem)
        Some(updatedMenuItem)
      }
      case None => None
    }
  }

  /**
   * Updates a MenuItem in the database.
   *
   * @param menuItem the MenuItem to update in the database
   * @return a future optional MenuItem depending on whether the operation was successful
   */
  def update(menuItem: MenuItem): Future[Option[MenuItem]] = {
    MenuItemTDG.update(
      UUID(menuItem.id).bytes,
      menuItem.version,
      menuItem.parentId,
      menuItem.tagLineId,
      menuItem.contentId,
      menuItem.language,
      menuItem.position,
      menuItem.title,
      menuItem.description,
      menuItem.longDescription,
      menuItem.published
    ).map {
      case Some(newVersion) => {
        val updatedMenuItem = menuItem.copy(version = newVersion)
        Cache.set("menuItem." + updatedMenuItem.id, updatedMenuItem)
        Some(updatedMenuItem)
      }
      case None => None
    }
  }

  /**
   * Deletes a MenuItem from the database.
   *
   * @param id id of the MenuItem to delete
   * @return a future boolean depending on whether the operation was successful
   */
  def delete(id: String): Future[Boolean] = {
    MenuItemTDG.delete(UUID(id).bytes)
  }

  def deleteCascade(id: String): Future[Boolean] = {
    MenuItemTDG.deleteCascade(UUID(id).bytes)
  }

  /**
   * Converts a RowData object into a MenuItem.
   *
   * @param row the row data to convert
   * @return a MenuItem object
   */
  private def rowToMenuItem(row: RowData): MenuItem = {
    val parentId = row("parent_id") match {
      case _: Array[Byte] => UUID((row("parent_id").asInstanceOf[Array[Byte]])).string
      case null | "" => ""
    }
    val tagLineId = row("tagline_id") match {
      case _: Array[Byte] => UUID((row("tagline_id").asInstanceOf[Array[Byte]])).string
      case null | "" => ""
    }
    val contentId = row("content_id") match {
      case _: Array[Byte] => UUID((row("content_id").asInstanceOf[Array[Byte]])).string
      case null | "" => ""
    }

    MenuItem(
      id              = UUID((row("id").asInstanceOf[Array[Byte]])).string,
      version         = row("version").asInstanceOf[Long],
      parentId        = Some(parentId),
      tagLineId       = Some(tagLineId),
      contentId       = Some(contentId),
      language        = row("language").asInstanceOf[String],
      position        = row("position").asInstanceOf[Int],
      title           = row("title").asInstanceOf[String],
      description     = row("description").asInstanceOf[String],
      longDescription = row("long_description").asInstanceOf[String],
      published       = row("published").asInstanceOf[Int]
    )
  }

}
