package models

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports.DateTime

import play.api.cache.Cache
import play.api.libs.json.{Writes, JsValue, Json}
import play.api.libs.functional.syntax._
import play.api.Play.current

import scala.concurrent.Future

import services.datasource.SelectFeaturedTDG

import helpers.UUID

case class SelectFeatured(
	id: UUID,
	translated_id: UUID,
	version: Long = 0,
	title: String,
	slug: String,
	description: String,
	href_slug: String,
	position: Integer,
	icon_type: String,
	content_type: String = "select_featured",
	primary_tag: String,
	starred: Integer = 9,
	status: Integer = 1,
	createdAt: Option[DateTime] = None,
	updatedAt: Option[DateTime] = None
) extends Content {

	override def duplicate(): Content = {
  	this.copy()
  }
}

case class SelectFeaturedFormData(
  var english_title: String,
  var french_title: String,
  var english_description: String,
  var french_description: String,
  var href_slug: String,
  var position: Int,
  var icon_type: String,
  var english_primary_tag: String,
  var french_primary_tag: String,
  var starred: Int)

object SelectFeatured {

  implicit val selectFeaturedWrites = new Writes[SelectFeatured] {
    def writes(selectFeatured: SelectFeatured): JsValue = {
      Json.obj(
        "id" -> selectFeatured.id.string,
        "translated_id" -> selectFeatured.translated_id.string,
        "title" -> selectFeatured.title,
        "description" -> selectFeatured.description,
        "select_featured_position" -> selectFeatured.position.asInstanceOf[Int],
        "select_featured_href_slug" -> selectFeatured.href_slug,
        "select_featured_icon_type" -> selectFeatured.icon_type,
        "content_type" -> selectFeatured.content_type,
        "primary_tag" -> selectFeatured.primary_tag,
        "starred" -> selectFeatured.starred.asInstanceOf[Int],
        "created_at" -> selectFeatured.createdAt
      )
    }
  }

	/**
	 * Find all selectFeatureds.
	 *
	 */
  def findAll: Future[Seq[SelectFeatured]] = {
    SelectFeaturedTDG.list.map { resultSet =>
      resultSet.map {
        item => rowToSelectFeatured(item)
      }
    }
  }

  def findByLanguage(language: String): Future[Seq[SelectFeatured]] = {
    SelectFeaturedTDG.findByLanguage(language).map { resultSet =>
      resultSet.map {
        item => rowToSelectFeatured(item)
      }
    }
  }

	/**
	 * @param selectFeatured the selectFeatured to create in the database
	 * @return an optional selectFeatured depending on whether the operation was successful
	 */
	def create(selectFeatured: SelectFeatured, language: String, user: User): Future[Option[SelectFeatured]] = {
		SelectFeaturedTDG.insert(
			selectFeatured.id.bytes,
			selectFeatured.translated_id.bytes,
			selectFeatured.title,
			selectFeatured.slug,
			selectFeatured.description,
			selectFeatured.href_slug,
			selectFeatured.position,
			selectFeatured.icon_type,
			selectFeatured.primary_tag,
			selectFeatured.starred,
			selectFeatured.content_type,
			language,
			user.id.bytes
		).map {
			case Some(newVersion) => {
				val updatedSelectFeatured = selectFeatured.copy(version = newVersion)
				Cache.set("selectFeatured." + updatedSelectFeatured.id, updatedSelectFeatured)
				Some(updatedSelectFeatured)
			}
			case _ => None
		}
	}

  def update(selectFeatured: SelectFeatured, user: User): Future[Option[SelectFeatured]] = {
    SelectFeaturedTDG.update(
			selectFeatured.id.bytes,
			selectFeatured.title,
			selectFeatured.slug,
			selectFeatured.description,
			selectFeatured.href_slug,
			selectFeatured.position,
			selectFeatured.icon_type,
			selectFeatured.primary_tag,
			selectFeatured.starred,
			user.id.bytes
    ).map {
			case Some(newVersion) => {
				val updatedSelectFeatured = selectFeatured.copy(version = newVersion)
				Cache.set("selectFeatured" + updatedSelectFeatured.id, updatedSelectFeatured)
				Some(updatedSelectFeatured)
			}
      case _ => None
    }
  }

	/**
	 * Converts a RowData object into a SelectFeatured.
	 *
	 * @param row the row data to convert
	 * @return a selectFeatured object
	 */
	def rowToSelectFeatured(row: RowData): SelectFeatured = {
		SelectFeatured(
			id           					= UUID(row("id").asInstanceOf[Array[Byte]]),
			translated_id 				= UUID(row("translated_id").asInstanceOf[Array[Byte]]),
			version     				  = row("version").asInstanceOf[Long],
			title 								= row("title").asInstanceOf[String],
			slug 									= UUID.random.string,
			description 				  = row("description").asInstanceOf[String],
			href_slug 	 					= row("select_featured_href_slug").asInstanceOf[String],
			position 							= row("select_featured_position").asInstanceOf[Integer],
			icon_type 						= row("select_featured_icon_type").asInstanceOf[String],
			content_type 				  = row("content_type").asInstanceOf[String],
			primary_tag 					= row("primary_tag").asInstanceOf[String],
			starred 							= row("starred").asInstanceOf[Integer],
			status 								= row("status").asInstanceOf[Integer],
			createdAt   					= Some(row("created_at").asInstanceOf[DateTime]),
			updatedAt   					= Some(row("updated_at").asInstanceOf[DateTime])
		)
	}

}
