package models

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports._

import play.api.cache.Cache
import play.api.libs.json.{Writes, JsValue, Json}
import play.api.libs.functional.syntax._
import play.api.Play.current

import scala.concurrent.Future

import services.datasource.SiteTDG

import helpers.UUID

case class Site(
  id: UUID,
  translated_id: UUID,
  version: Long = 0,
  title: String,
  slug: String,
  description: String,
  content_type: String = "site",
  primary_tag: String,
  starred: Integer = 9,
  site_logo: String,
  content: String,
  site_url: String,
  status: Integer = 1,
  createdAt: Option[DateTime] = None,
  updatedAt: Option[DateTime] = None
) extends Content {

  override def duplicate(): Content = {
    this.copy()
  }
}

case class SiteFormData(
  var english_title: String,
  var french_title: String,
  var english_description: String,
  var french_description: String,
  var english_content: String,
  var french_content: String,
  var english_site_url: String,
  var french_site_url: String,
  var english_tags: String,
  var french_tags: String,
  var starred: Int)

object Site {

  /**
   * Find all sites.
   *
   */
  def findAll: Future[Seq[Site]] = {
    SiteTDG.list.map { resultSet =>
      resultSet.map {
        item => rowToSite(item)
      }
    }
  }

  def findByLanguage(language: String): Future[Seq[Site]] = {
    SiteTDG.findByLanguage(language).map { resultSet =>
      resultSet.map {
        item => rowToSite(item)
      }
    }
  }

  /**
   * @param site the site to create in the database
   * @return an optional site depending on whether the operation was successful
   */
  def create(site: Site, language: String, user: User): Future[Option[Site]] = {
    val urlchars = "abcdefghijklmnopqrstuvwxyz0123456789-_.~/"
    val buf = new StringBuilder
    val s = site.slug.toLowerCase
    for (c <- s) if (urlchars contains c) buf.append( c ) else if (c == ' ') buf.append( '-' )

    SiteTDG.insert(
      site.id.bytes,
      site.translated_id.bytes,
      site.title,
      buf.toString,
      site.description,
      site.content_type,
      site.primary_tag,
      site.starred,
      site.site_logo,
      site.content,
      site.site_url,
      language,
      user.id.bytes
    ).map {
      case Some(newVersion) => {
        val updatedSite = site.copy(version = newVersion)
        Cache.set("site." + updatedSite.id, updatedSite)
        Some(updatedSite)
      }
      case _ => None
    }
  }

  def update(site: Site, user: User): Future[Option[Site]] = {
    val urlchars = "abcdefghijklmnopqrstuvwxyz0123456789-_.~/"
    val buf = new StringBuilder
    val s = site.slug.toLowerCase
    for (c <- s) if (urlchars contains c) buf.append( c ) else if (c == ' ') buf.append( '-' )

    SiteTDG.update(
      site.id.bytes,
      site.title,
      buf.toString,
      site.description,
      site.site_logo,
      site.content,
      site.site_url,
      site.primary_tag,
      site.starred,
      user.id.bytes
    ).map {
      case Some(newVersion) => {
        val updatedSite = site.copy(version = newVersion)
        Cache.set("site." + updatedSite.id, updatedSite)
        Some(updatedSite)
      }
      case _ => None
    }
  }

  /**
   * Converts a RowData object into a Site.
   *
   * @param row the row data to convert
   * @return a site object
   */
  def rowToSite(row: RowData): Site = {
    Site(
      id                    = UUID(row("id").asInstanceOf[Array[Byte]]),
      translated_id         = UUID(row("translated_id").asInstanceOf[Array[Byte]]),
      version               = row("version").asInstanceOf[Long],
      title                 = row("title").asInstanceOf[String],
      slug                  = row("slug").asInstanceOf[String],
      description           = row("description").asInstanceOf[String],
      content_type          = row("content_type").asInstanceOf[String],
      primary_tag           = row("primary_tag").asInstanceOf[String],
      starred               = row("starred").asInstanceOf[Integer],
      site_logo             = row("site_logo").asInstanceOf[String],
      content               = row("site_content").asInstanceOf[String],
      site_url              = row("site_url").asInstanceOf[String],
      status                = row("status").asInstanceOf[Integer],
      createdAt             = Some(row("created_at").asInstanceOf[DateTime]),
      updatedAt             = Some(row("updated_at").asInstanceOf[DateTime])
    )
  }

}
