package models

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports.DateTime

import play.api.cache.Cache
import play.api.libs.json.{Writes, JsValue, Json}
import play.api.libs.functional.syntax._
import play.api.Play.current

import scala.concurrent.Future

import services.datasource.SiteWordTDG

import helpers.UUID

case class SiteWord(
	id: UUID,
	translated_id: UUID,
	version: Long = 0,
	title: String,
	slug: String,
	description: String,
	href_id: UUID,
	position: Integer,
	content_type: String = "site_word",
	primary_tag: String = "site_word",
	starred: Integer = 9,
	status: Integer = 1,
	createdAt: Option[DateTime] = None,
	updatedAt: Option[DateTime] = None
) extends Content {

	override def duplicate(): Content = {
  	this.copy()
  }
}

case class SiteWordFormData(
  var english_title: String,
  var french_title: String,
  var english_description: String,
  var french_description: String,
  var href_id: String,
  var position: Int)

object SiteWord {

	/**
	 * Find all siteWords.
	 *
	 */
  def findAll: Future[Seq[SiteWord]] = {
    SiteWordTDG.list.map { resultSet =>
      resultSet.map {
        item => rowToSiteWord(item)
      }
    }
  }

  def findByLanguage(language: String): Future[Seq[SiteWord]] = {
    SiteWordTDG.findByLanguage(language).map { resultSet =>
      resultSet.map {
        item => rowToSiteWord(item)
      }
    }
  }

  def setPosition(id: String, position: Int): Future[Boolean] = {
    SiteWordTDG.setPosition(UUID(id).bytes, position)
  }

	/**
	 * @param siteWord the siteWord to create in the database
	 * @return an optional siteWord depending on whether the operation was successful
	 */
	def create(siteWord: SiteWord, language: String, user: User): Future[Option[SiteWord]] = {
		SiteWordTDG.insert(
			siteWord.id.bytes,
			siteWord.translated_id.bytes,
			siteWord.title,
			siteWord.slug,
			siteWord.description,
			siteWord.href_id.bytes,
			siteWord.position,
			siteWord.content_type,
			siteWord.primary_tag,
			language,
			user.id.bytes
		).map {
			case Some(newVersion) => {
				val updatedSiteWord = siteWord.copy(version = newVersion)
				Cache.set("siteWord." + updatedSiteWord.id, updatedSiteWord)
				Some(updatedSiteWord)
			}
			case _ => None
		}
	}

  def update(siteWord: SiteWord, user: User): Future[Option[SiteWord]] = {
    SiteWordTDG.update(
			siteWord.id.bytes,
			siteWord.title,
			siteWord.slug,
			siteWord.description,
			siteWord.href_id.bytes,
			siteWord.position,
			siteWord.primary_tag,
			user.id.bytes
    ).map {
			case Some(newVersion) => {
				val updatedSiteWord = siteWord.copy(version = newVersion)
				Cache.set("siteWord" + updatedSiteWord.id, updatedSiteWord)
				Some(updatedSiteWord)
			}
      case _ => None
    }
  }

	/**
	 * Converts a RowData object into a SiteWord.
	 *
	 * @param row the row data to convert
	 * @return a siteWord object
	 */
	def rowToSiteWord(row: RowData): SiteWord = {
		SiteWord(
			id           					= UUID(row("id").asInstanceOf[Array[Byte]]),
			translated_id 				= UUID(row("translated_id").asInstanceOf[Array[Byte]]),
			version     				  = row("version").asInstanceOf[Long],
			title 								= row("title").asInstanceOf[String],
			slug 									= UUID.random.string,
			description 				  = row("description").asInstanceOf[String],
			href_id 	 					  = UUID(row("site_word_href_id").asInstanceOf[Array[Byte]]),
			position 							= row("site_word_position").asInstanceOf[Integer],
			content_type 				  = row("content_type").asInstanceOf[String],
			primary_tag 					= row("primary_tag").asInstanceOf[String],
			status 								= row("status").asInstanceOf[Integer],
			createdAt   					= Some(row("created_at").asInstanceOf[DateTime]),
			updatedAt   					= Some(row("updated_at").asInstanceOf[DateTime])
		)
	}

}
