package models

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports._

import play.api.cache.Cache
import play.api.libs.json.{Writes, JsValue, Json}
import play.api.libs.functional.syntax._
import play.api.Play.current

import scala.concurrent.Future

import services.datasource.SlideTDG

import helpers.UUID

case class Slide(
  id: UUID,
  translated_id: UUID,
  version: Long = 0,
  title: String,
  slug: String,
  description: String,
  position: Integer,
  content_type: String = "slide",
  primary_tag: String = "slide",
  starred: Integer = 9,
  slide_background: String,
  content: String = "",
  slide_url: String,
  header_colour: String = "white",
  status: Integer = 1,
  createdAt: Option[DateTime] = None,
  updatedAt: Option[DateTime] = None
) extends Content {

  override def duplicate(): Content = {
    this.copy()
  }
}

case class SlideFormData(
  english_title: String,
  french_title: String,
  english_description: String,
  french_description: String,
  position: Int,
  english_slide_url: String,
  french_slide_url: String,
  header_colour: String
)

object Slide {

  /**
   * Find all slides.
   *
   */
  def findAll: Future[Seq[Slide]] = {
    SlideTDG.list.map { resultSet =>
      resultSet.map {
        item => rowToSlide(item)
      }
    }
  }

  def findByLanguage(language: String): Future[Seq[Slide]] = {
    SlideTDG.findByLanguage(language).map { resultSet =>
      resultSet.map {
        item => rowToSlide(item)
      }
    }
  }

  /**
   * @param slide the slide to create in the database
   * @return an optional slide depending on whether the operation was successful
   */
  def create(slide: Slide, language: String, user: User): Future[Option[Slide]] = {
    SlideTDG.insert(
      slide.id.bytes,
      slide.translated_id.bytes,
      slide.title,
      slide.slug,
      slide.description,
      slide.position,
      slide.content_type,
      slide.slide_background,
      slide.content,
      slide.slide_url,
      slide.header_colour,
      language,
      user.id.bytes
    ).map {
      case Some(newVersion) => {
        val updatedSlide = slide.copy(version = newVersion)
        Cache.set("slide." + updatedSlide.id, updatedSlide)
        Some(updatedSlide)
      }
      case _ => None
    }
  }

  def update(slide: Slide, user: User): Future[Option[Slide]] = {
    SlideTDG.update(
      slide.id.bytes,
      slide.title,
      slide.slug,
      slide.description,
      slide.position,
      slide.slide_background,
      slide.content,
      slide.slide_url,
      slide.header_colour,
      user.id.bytes
    ).map {
      case Some(newVersion) => {
        val updatedSlide = slide.copy(version = newVersion)
        Cache.set("slide." + updatedSlide.id, updatedSlide)
        Some(updatedSlide)
      }
      case _ => None
    }
  }

  /**
   * Converts a RowData object into a Slide.
   *
   * @param row the row data to convert
   * @return a slide object
   */
  def rowToSlide(row: RowData): Slide = {
    Slide(
      id                    = UUID(row("id").asInstanceOf[Array[Byte]]),
      translated_id         = UUID(row("translated_id").asInstanceOf[Array[Byte]]),
      version               = row("version").asInstanceOf[Long],
      title                 = row("title").asInstanceOf[String],
      slug                  = row("slug").asInstanceOf[String],
      description           = row("description").asInstanceOf[String],
      position              = row("slide_position").asInstanceOf[Integer],
      content_type          = row("content_type").asInstanceOf[String],
      slide_background      = row("slide_background").asInstanceOf[String],
      content               = row("slide_content").asInstanceOf[String],
      slide_url             = row("slide_url").asInstanceOf[String],
      header_colour         = row("header_colour").asInstanceOf[String],
      status                = row("status").asInstanceOf[Integer],
      createdAt             = Some(row("created_at").asInstanceOf[DateTime]),
      updatedAt             = Some(row("updated_at").asInstanceOf[DateTime])
    )
  }

}
