package models

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports.DateTime

import play.api.cache.Cache
import play.api.libs.json.{Writes, JsValue, Json}
import play.api.libs.functional.syntax._
import play.api.Play.current

import scala.concurrent.duration._
import scala.concurrent.{Future, Await}

import services.datasource.TagTDG

import helpers.UUID

case class Tag(
	id: UUID = UUID.random,
	version: Long = 0,
	name: String,
	count: Long = 0,
	createdAt: Option[DateTime] = None,
	updatedAt: Option[DateTime] = None
)

case class TagFormData(
  var name: String)

object Tag {

	/**
	 * Find single tag.
	 *
	 * @param id the UUID of the tag
	 * @return an optional tag
	 */
	def find(id: UUID): Future[Option[Tag]] = {
    TagTDG.find(id.bytes).map { result =>
    	result match {
    		case IndexedSeq(item) => Some(rowToTag(item))
    		case _ => None
    	}
    }
	}

	/**
	 * Find single tag.
	 *
	 * @param name the name of the tag
	 * @return an optional tag
	 */
	def findByName(name: String, language: String): Future[Option[Tag]] = {
		TagTDG.findByName(name, language).map { result =>
			result match {
				case IndexedSeq(item) => Some(rowToTag(item))
				case _ => None
			}
		}
	}

	/**
	 * Find tags by language.
	 *
	 */
	def findByLanguage(language: String): Future[Seq[Tag]] = {
		TagTDG.findByLanguage(language).map { resultSet =>
			resultSet.map {
				item => rowToTag(item)
			}
		}
	}

	/**
	 * Find all tags.
	 *
	 */
  def findAll: Future[Seq[Tag]] = {
    TagTDG.list.map { resultSet =>
      resultSet.map {
        item => rowToTag(item)
      }
    }
  }

  /**
   * Find all tags by content id.
   *
   */
  def findByTagged(content_id: Array[Byte]): Future[Seq[Tag]] = {
  	TagTDG.findByTagged(content_id).map { resultSet =>
  		resultSet.map {
  			item => rowToTag(item)
  		}
  	}
  }

  def findByManyTagged(content_ids: String): Future[Seq[Tag]] = {
  	var contentIdArray = content_ids.split(",").distinct.map( id => UUID(id.trim).bytes )

  	TagTDG.findByManyTagged(contentIdArray).map { resultSet =>
  		resultSet.map {
  			item => rowToTag(item)
  		}
  	}
  }

  def findByTags(tags: String, language: String): Future[Seq[Tag]] = {
    var tagArray = tags.split(",").distinct.map( tag => tag.trim )

    TagTDG.findByTags(tagArray, language).map { resultSet =>
      resultSet.map {
        item => rowToTag(item)
      }
    }
  }

	/**
	 * @param tag the tag to create in the database
	 * @return an optional tag depending on whether the operation was successful
	 */
	def create(tag: Tag, language: String): Future[Option[Tag]] = {
		TagTDG.insert(
			tag.id.bytes,
			tag.name,
			language
		).map {
			case Some(newVersion) => {
				val updatedTag = tag.copy(version = newVersion)
				Cache.set("tag." + updatedTag.id, updatedTag)
				Some(updatedTag)
			}
			case _ => None
		}
	}

  def update(tag: Tag): Future[Option[Tag]] = {
    TagTDG.update(
			tag.id.bytes,
			tag.name
    ).map {
			case Some(newVersion) => {
				val updatedTag = tag.copy(version = newVersion)
				Cache.set("tag" + updatedTag.id, updatedTag)
				Some(updatedTag)
			}
      case _ => None
    }
  }

	def tagContent(tags: String, content_id: Array[Byte], language: String) = {
		Tag.untagContent(content_id)

		var tagArray = tags.split(",").distinct.map( tag => tag.trim).filterNot(_ matches (" *"))

		for (i <- 0 to (tagArray.length - 1)) {
	    var futureFind = Tag.findByName(tagArray(i), language)
			var foundTag = Await.result(futureFind, 5 seconds)

			if (foundTag != None) Await.result(TagTDG.tagContent(foundTag.get.id.bytes, content_id), 5 seconds)
			else {
				var futureTag = Tag.create(new Tag(name = tagArray(i)), language)
				var createdTag = Await.result(futureTag, 5 seconds)

				if (createdTag != None)	Await.result(TagTDG.tagContent(createdTag.get.id.bytes, content_id), 5 seconds)
			}
		}
	}

	def untagContent(content_id: Array[Byte]) = {
		Await.result(TagTDG.untagContent(content_id), 5 seconds)
	}

  def delete(id: UUID): Future[Boolean] = {
    TagTDG.delete(id.bytes)
  }

	/**
	 * Converts a RowData object into a Tag.
	 *
	 * @param row the row data to convert
	 * @return a tag object
	 */
	private def rowToTag(row: RowData): Tag = {
		Tag(
			id           					= UUID(row("id").asInstanceOf[Array[Byte]]),
			version     				  = row("version").asInstanceOf[Long],
			name 									= row("name").asInstanceOf[String],
			count 								= row("count").asInstanceOf[Long],
			createdAt   					= Some(row("created_at").asInstanceOf[DateTime]),
			updatedAt   					= Some(row("updated_at").asInstanceOf[DateTime])
		)
	}

}
