package models

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports.DateTime

import play.api.cache.Cache
import play.api.libs.json.{Writes, JsValue, Json}
import play.api.libs.functional.syntax._
import play.api.Play.current

import scala.concurrent.Future

import services.datasource.TagLineTDG

import helpers.UUID

case class TagLine(
	id: UUID,
	translated_id: UUID,
	version: Long = 0,
	title: String,
	slug: String,
	description: String,
	position: Integer,
	content_type: String = "tag_line",
	primary_tag: String = "tag_line",
	starred: Integer = 9,
	status: Integer = 1,
	createdAt: Option[DateTime] = None,
	updatedAt: Option[DateTime] = None
) extends Content {

	override def duplicate(): Content = {
  	this.copy()
  }
}

case class TagLineFormData(
  var english_title: String,
  var french_title: String,
  var english_description: String,
  var french_description: String,
  var position: Int,
  var english_tags: String,
  var french_tags: String)

object TagLine {

  implicit val tagLineWrites = new Writes[TagLine] {
    def writes(tagLine: TagLine): JsValue = {
      Json.obj(
        "id" -> tagLine.id.string,
        "translated_id" -> tagLine.translated_id.string,
        "title" -> tagLine.title,
        "description" -> tagLine.description,
        "content_type" -> tagLine.content_type,
        "created_at" -> tagLine.createdAt
      )
    }
  }

	/**
	 * Find single tag_line.
	 *
	 * @param id the UUID of the tag_line
	 * @return an optional tag_line
	 */
	def find(id: UUID): Future[Option[TagLine]] = {
    TagLineTDG.find(id.bytes).map { result =>
    	result match {
    		case IndexedSeq(item) => Some(rowToTagLine(item))
    		case _ => None
    	}
    }
	}

	/**
	 * Find all tag_lines.
	 *
	 */
  def findAll: Future[Seq[TagLine]] = {
    TagLineTDG.list.map { resultSet =>
      resultSet.map {
        item => rowToTagLine(item)
      }
    }
  }

  def findByLanguage(language: String): Future[Seq[TagLine]] = {
    TagLineTDG.findByLanguage(language).map { resultSet =>
      resultSet.map {
        item => rowToTagLine(item)
      }
    }
  }

  def setPosition(id: String, position: Int): Future[Boolean] = {
    TagLineTDG.setPosition(UUID(id).bytes, position)
  }

	/**
	 * @param tagLine the tagLine to create in the database
	 * @return an optional tagLine depending on whether the operation was successful
	 */
	def create(tagLine: TagLine, language: String, user: User): Future[Option[TagLine]] = {
		TagLineTDG.insert(
			tagLine.id.bytes,
			tagLine.translated_id.bytes,
			tagLine.title,
			tagLine.slug,
			tagLine.description,
			tagLine.position,
			tagLine.content_type,
			language,
			user.id.bytes
		).map {
			case Some(newVersion) => {
				val updatedTagLine = tagLine.copy(version = newVersion)
				Cache.set("tagLine." + updatedTagLine.id, updatedTagLine)
				Some(updatedTagLine)
			}
			case _ => None
		}
	}

  def update(tagLine: TagLine, user: User): Future[Option[TagLine]] = {
    TagLineTDG.update(
			tagLine.id.bytes,
			tagLine.title,
			tagLine.slug,
			tagLine.description,
			tagLine.position,
			user.id.bytes
    ).map {
			case Some(newVersion) => {
				val updatedTagLine = tagLine.copy(version = newVersion)
				Cache.set("tagLine" + updatedTagLine.id, updatedTagLine)
				Some(updatedTagLine)
			}
      case _ => None
    }
  }

	/**
	 * Converts a RowData object into a TagLine.
	 *
	 * @param row the row data to convert
	 * @return a tag_line object
	 */
	def rowToTagLine(row: RowData): TagLine = {
		TagLine(
			id           					= UUID(row("id").asInstanceOf[Array[Byte]]),
			translated_id 				= UUID(row("translated_id").asInstanceOf[Array[Byte]]),
			version     				  = row("version").asInstanceOf[Long],
			title 								= row("title").asInstanceOf[String],
			slug 									= UUID.random.string,
			description 				  = row("description").asInstanceOf[String],
			position 							= row("tag_line_position").asInstanceOf[Integer],
			content_type 				  = row("content_type").asInstanceOf[String],
			status 								= row("status").asInstanceOf[Integer],
			createdAt   					= Some(row("created_at").asInstanceOf[DateTime]),
			updatedAt   					= Some(row("updated_at").asInstanceOf[DateTime])
		)
	}

}
