package models

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports._

import play.api.cache.Cache
import play.api.libs.json.{Writes, JsValue, Json}
import play.api.libs.functional.syntax._
import play.api.Play.current

import scala.concurrent.Future

import services.datasource.TextTDG

import helpers.UUID

case class Text(
  id: UUID,
  translated_id: UUID,
  version: Long = 0,
  title: String,
  slug: String,
  description: String,
  content_type: String = "text",
  primary_tag: String,
  starred: Integer = 9,
  content: String,
  status: Integer = 1,
  createdAt: Option[DateTime] = None,
  updatedAt: Option[DateTime] = None
) extends Content {

  override def duplicate(): Content = {
    this.copy()
  }
}

case class TextFormData(
  var english_title: String,
  var french_title: String,
  var english_slug: String,
  var french_slug: String,
  var english_description: String,
  var french_description: String,
  var english_content: String,
  var french_content: String,
  var english_primary_tag: String,
  var french_primary_tag: String,
  var english_tags: String,
  var french_tags: String,
  var starred: Int)

object Text {

  implicit val textWrites = new Writes[Text] {
    def writes(text: Text): JsValue = {
      Json.obj(
        "id" -> text.id.string,
         "translated_id" -> text.translated_id.string,
         "title"  -> text.title,
         "slug" -> text.slug,
         "description" -> text.description,
         "content_type" -> text.content_type,
         "primary_tag" -> text.primary_tag,
         "starred" -> text.starred.asInstanceOf[Int],
         "content" -> text.content,
         "created_at" -> text.createdAt
      )
    }
  }

  /**
   * Find all texts.
   *
   */
  def findAll: Future[Seq[Text]] = {
    TextTDG.list.map { resultSet =>
      resultSet.map {
        item => rowToText(item)
      }
    }
  }

  /**
   * @param text the text to create in the database
   * @return an optional text depending on whether the operation was successful
   */
  def create(text: Text, language: String, user: User): Future[Option[Text]] = {
    val urlchars = "abcdefghijklmnopqrstuvwxyz0123456789-_.~/"
    val buf = new StringBuilder
    val s = text.slug.toLowerCase
    for (c <- s) if (urlchars contains c) buf.append( c ) else if (c == ' ') buf.append( '-' )

    TextTDG.insert(
      text.id.bytes,
      text.translated_id.bytes,
      text.title,
      buf.toString,
      text.description,
      text.content_type,
      text.primary_tag,
      text.starred,
      text.content,
      language,
      user.id.bytes
    ).map {
      case Some(newVersion) => {
        val updatedText = text.copy(version = newVersion)
        Cache.set("text." + updatedText.id, updatedText)
        Some(updatedText)
      }
      case _ => None
    }
  }

  def update(text: Text, user: User): Future[Option[Text]] = {
    val urlchars = "abcdefghijklmnopqrstuvwxyz0123456789-_.~/"
    val buf = new StringBuilder
    val s = text.slug.toLowerCase
    for (c <- s) if (urlchars contains c) buf.append( c ) else if (c == ' ') buf.append( '-' )

    TextTDG.update(
      text.id.bytes,
      text.title,
      buf.toString,
      text.description,
      text.content,
      text.primary_tag,
      text.starred,
      user.id.bytes
    ).map {
      case Some(newVersion) => {
        val updatedText = text.copy(version = newVersion)
        Cache.set("text." + updatedText.id, updatedText)
        Some(updatedText)
      }
      case _ => None
    }
  }

  /**
   * Converts a RowData object into a Text.
   *
   * @param row the row data to convert
   * @return a text object
   */
  def rowToText(row: RowData): Text = {
    Text(
      id              = UUID(row("id").asInstanceOf[Array[Byte]]),
      translated_id   = UUID(row("translated_id").asInstanceOf[Array[Byte]]),
      version         = row("version").asInstanceOf[Long],
      title           = row("title").asInstanceOf[String],
      slug            = row("slug").asInstanceOf[String],
      description     = row("description").asInstanceOf[String],
      content_type    = row("content_type").asInstanceOf[String],
      primary_tag     = row("primary_tag").asInstanceOf[String],
      starred         = row("starred").asInstanceOf[Integer],
      content         = row("content").asInstanceOf[String],
      status          = row("status").asInstanceOf[Integer],
      createdAt       = Some(row("created_at").asInstanceOf[DateTime]),
      updatedAt       = Some(row("updated_at").asInstanceOf[DateTime])
    )
  }

}
