package models

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports._

import play.api.cache.Cache
import play.api.libs.json.{Writes, JsValue, Json}
import play.api.libs.functional.syntax._
import play.api.Play.current

import scala.concurrent.Future

import services.datasource.UserTDG

import helpers.UUID

case class User(
  id: UUID = UUID.random,
  version: Long = 0,
  email: String,
  password: String,
  status: Integer = 1,
  createdAt: Option[DateTime] = None,
  updatedAt: Option[DateTime] = None
) extends Model {

  def create: Future[Option[User]] = {
    User.create(this)
  }

  def update: Future[Option[User]] = {
    User.update(this)
  }

  def delete: Future[Option[User]] = {
    User.delete(this)
  }

  // def listRoles = {
  //   User.listRoles(this)
  // }

  // def roles: Future[IndexedSeq[String]] = {
  //   Cache.getAs[IndexedSeq[Role]]("userRoles." + id) match {
  //     case Some(roles) => Future(roles.map(_.toString().toLowerCase()))
  //     case None => listRoles.map { roles =>
  //       roles.map(_.toString().toLowerCase())
  //     }
  //   }
  // }

}

/**
 * User companion object. Fills the role of input/output mappers in the
 * layered architecture. This object should handle all interaction with the
 * technical services layer for Users.
 *
 * For individual user lookups, this mapper will check the in-memory cache
 * first, followed by the database.
 */
object User {

  // implicit val userReads = (
  //   (__ \ "id").read[String] and
  //   (__ \ "version").read[Long] and
  //   (__ \ "email").read[String] and
  //   (__ \ "givenname").read[String] and
  //   (__ \ "surname").read[String] and
  //   (__ \ "organization").read[String] and
  //   (__ \ "school").read[String] and
  //   (__ \ "mailingPref").read[String] and
  //   (__ \ "createdAt").readNullable[DateTime] and
  //   (__ \ "updatedAt").readNullable[DateTime]
  // )(User.apply(_: String, _: Long, _: String, _: String, _: String, _: String, _: String, _: String, _: Option[DateTime], _: Option[DateTime]))

  // implicit val userWrites = new Writes[User] {
  //   def writes(user: User): JsValue = {
  //     Json.obj(
  //       "id" -> user.id.string,
  //       "version" -> user.version,
  //       "email" -> user.email,
  //       "givenname" -> user.givenname,
  //       "surname" -> user.surname,
  //       "organization" -> user.organization,
  //       "school" -> user.school,
  //       "mailingPref" -> user.mailingPref
  //     )
  //   }
  // }

  /**
   * Converts RowData object returned from the database into a User domain model.
   */
  private def rowToUser(row: RowData): User = {
    User(
      id           = UUID(row("id").asInstanceOf[Array[Byte]]),
      version      = row("version").asInstanceOf[Long],
      email        = row("email").asInstanceOf[String],
      password     = row("password").asInstanceOf[String],
      status       = row("status").asInstanceOf[Integer],
      createdAt    = Some(row("created_at").asInstanceOf[DateTime]),
      updatedAt    = Some(row("updated_at").asInstanceOf[DateTime])
    )
  }

  /**
   * Find all users.
   */
  def findAll: Future[Seq[User]] = {
    UserTDG.list.map { resultSet =>
      resultSet.map {
        item => rowToUser(item)
      }
    }
  }

  def authenticate(email: String, password: String): Future[Option[User]] = {
    val futureUser = UserTDG.authenticate(email, password)

    for {
      maybeUser <- futureUser
    } yield {
      maybeUser match {
        case Some(rowData) => {
          val user = rowToUser(rowData)
          Cache.set("user." + user.id.string, user)
          Some(user)
        }
        case _ => None
      }
    }
  }

  /**
   * Finds a single user by ID.
   *
   * Checks in the cache before hitting the database. Once
   * a user is found, this sets it into the cache.
   *
   * @param id
   * @param ctx
   * @return a promise of maybe a user
   */
  def find(id: String): Future[Option[User]] = {
    Cache.getAs[User]("user." + id) match {
      case Some(user) => {
        Future.successful(Some(user))
      }
      case None => {
        UserTDG.find(UUID(id).bytes).map { result =>
          result match {
            case IndexedSeq(item) => Some(rowToUser(item))
            case _ => None
          }
        }
      }
    }
  }

  def findByEmail(email: String): Future[Option[User]] = {
    UserTDG.findByEmail(email).map { result =>
      result match {
        case IndexedSeq(item) => Some(rowToUser(item))
        case _ => None
      }
    }
  }

  // /**
  //  * Finds a single user by email address.
  //  *
  //  * This function always performs a database lookup and
  //  * is usually used for authentication.
  //  *
  //  * @param email
  //  * @param ctx
  //  * @return
  //  */
  // def findByEmail(email: String): Future[Option[User]] = {
  //   UserTDG.findByEmail(email).map { result =>
  //     result match {
  //       case IndexedSeq(item) => Some(rowToUser(item))
  //       case _ => None
  //     }
  //   }
  // }

  /**
   * Create a new user.
   */
  def create(user: User): Future[Option[User]] = {
    UserTDG.insert(
      user.id.bytes,
      user.version,
      user.email,
      user.password
    ).map {
      case Some(newVersion) => {
        val newUser = user.copy(version = newVersion)
        Cache.set("user." + newUser.id, newUser)
        Some(newUser)
      }
      case _ => None
    }
  }

  /**
   * Update an existing user.
   */
  def update(user: User): Future[Option[User]] = {
    UserTDG.update(
      user.id.bytes,
      user.version,
      user.email,
      user.password
    ).map {
      case Some(newVersion) => {
        val updatedUser = user.copy(version = newVersion)
        Cache.set("user." + updatedUser.id, updatedUser)
        Some(updatedUser)
      }
      case _ => None
    }
  }

  /**
   * Delete an existing user.
   */
  def delete(user: User): Future[Option[User]] = {
    UserTDG.delete(user.id.string, user.version).map {
      case true => {
        Cache.remove("user." + user.id.string)
        Some(user)
      }
      case _ => None
    }
  }
}
