package models

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports._

import play.api.cache.Cache
import play.api.libs.json.{Writes, JsValue, Json}
import play.api.libs.functional.syntax._
import play.api.Play.current

import scala.concurrent.Future

import services.datasource.VideoTDG

import helpers.UUID

case class Video(
  id: UUID,
  translated_id: UUID,
  version: Long = 0,
  title: String,
  slug: String,
  description: String,
  long_description: String,
  content_type: String = "video",
  primary_tag: String,
  starred: Integer = 9,
  video_type: String,
  video_url: String,
  status: Integer = 1,
  createdAt: Option[DateTime] = None,
  updatedAt: Option[DateTime] = None
) extends Content {

  override def duplicate(): Content = {
    this.copy()
  }
}

case class VideoFormData(
  var english_title: String,
  var french_title: String,
  var english_slug: String,
  var french_slug: String,
  var english_description: String,
  var french_description: String,
  var english_long_description: String,
  var french_long_description: String,
  var video_type: String,
  var english_video_url: String,
  var french_video_url: String,
  var english_primary_tag: String,
  var french_primary_tag: String,
  var english_tags: String,
  var french_tags: String,
  var starred: Int)

object Video {

  implicit val videoWrites = new Writes[Video] {
    def writes(video: Video): JsValue = {
      Json.obj(
        "id" -> video.id.string,
        "translated_id" -> video.translated_id.string,
        "title"  -> video.title,
        "slug" -> video.slug,
        "description" -> video.description,
        "long_description" -> video.long_description,
        "content_type" -> video.content_type,
        "primary_tag" -> video.primary_tag,
        "starred" -> video.starred.asInstanceOf[Int],
        "video_type" -> video.video_type,
        "video_url" -> video.video_url,
        "created_at" -> video.createdAt
      )
    }
  }

  /**
   * Find all videos.
   *
   */
  def findAll: Future[Seq[Video]] = {
    VideoTDG.list.map { resultSet =>
      resultSet.map {
        item => rowToVideo(item)
      }
    }
  }

  /**
   * @param video the video to create in the database
   * @return an optional video depending on whether the operation was successful
   */
  def create(video: Video, language: String, user: User): Future[Option[Video]] = {
    val urlchars = "abcdefghijklmnopqrstuvwxyz0123456789-_.~/"
    val buf = new StringBuilder
    val s = video.slug.toLowerCase
    for (c <- s) if (urlchars contains c) buf.append( c ) else if (c == ' ') buf.append( '-' )

    VideoTDG.insert(
      video.id.bytes,
      video.translated_id.bytes,
      video.title,
      buf.toString,
      video.description,
      video.long_description,
      video.content_type,
      video.primary_tag,
      video.starred,
      video.video_type,
      video.video_url,
      language,
      user.id.bytes
    ).map {
      case Some(newVersion) => {
        val updatedVideo = video.copy(version = newVersion)
        Cache.set("video." + updatedVideo.id, updatedVideo)
        Some(updatedVideo)
      }
      case _ => None
    }
  }

  def update(video: Video, user: User): Future[Option[Video]] = {
    val urlchars = "abcdefghijklmnopqrstuvwxyz0123456789-_.~/"
    val buf = new StringBuilder
    val s = video.slug.toLowerCase
    for (c <- s) if (urlchars contains c) buf.append( c ) else if (c == ' ') buf.append( '-' )

    VideoTDG.update(
      video.id.bytes,
      video.title,
      buf.toString,
      video.description,
      video.long_description,
      video.video_type,
      video.video_url,
      video.primary_tag,
      video.starred,
      user.id.bytes
    ).map {
      case Some(newVersion) => {
        val updatedVideo = video.copy(version = newVersion)
        Cache.set("video." + updatedVideo.id, updatedVideo)
        Some(updatedVideo)
      }
      case _ => None
    }
  }

  /**
   * Converts a RowData object into a Video.
   *
   * @param row the row data to convert
   * @return a video object
   */
  def rowToVideo(row: RowData): Video = {
    Video(
      id                     = UUID(row("id").asInstanceOf[Array[Byte]]),
      translated_id          = UUID(row("translated_id").asInstanceOf[Array[Byte]]),
      version                = row("version").asInstanceOf[Long],
      title                  = row("title").asInstanceOf[String],
      slug                   = row("slug").asInstanceOf[String],
      description            = row("description").asInstanceOf[String],
      long_description       = row("long_description").asInstanceOf[String],
      content_type           = row("content_type").asInstanceOf[String],
      primary_tag            = row("primary_tag").asInstanceOf[String],
      starred                = row("starred").asInstanceOf[Integer],
      video_type             = row("video_type").asInstanceOf[String],
      video_url              = row("video_url").asInstanceOf[String],
      status                 = row("status").asInstanceOf[Integer],
      createdAt              = Some(row("created_at").asInstanceOf[DateTime]),
      updatedAt              = Some(row("updated_at").asInstanceOf[DateTime])
    )
  }

}
