package services.datasource

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports._
import webcrank.password._
import scala.concurrent.Future
import org.joda.time.LocalDate
import play.api.Logger

object ContentTDG extends TableDataGateway {

  /*
  The SQL queries!

  Write all SQL queries that need for interacting with the contents table here.
 */

  val Insert = """
    INSERT INTO contents (id, translated_id, version, title, description, content_type, status, created_at, updated_at)
    VALUES (?, ?, 1, ?, ?, ?, 1, ?, ?)
    RETURNING version
  """

  val Update = """
    UPDATE contents
    SET version = ?, title = ?, description = ?, updated_at = ?
    WHERE id = ?
      AND status = 1
    RETURNING version
  """

  val SelectLanguage = """
  WITH content AS (
    SELECT title
    FROM contents
    WHERE id = ?
  ), translation AS (
    SELECT language_id
    FROM translations
    WHERE id = (SELECT title FROM content)
  )
  SELECT name AS language
  FROM languages
  WHERE languages.id = (SELECT language_id FROM translation)
  """

  val SelectSlug = """
  SELECT slug
  FROM contents
  WHERE id = ?
  """

  override val SelectAll = """
    SELECT  contents.id,
            contents.translated_id,
            contents.slug,
            contents.version,
            content_type,
            primary_tag,
            starred,
            contents.status,
            contents.created_at,
            contents.updated_at,
            video_contents.video_type,
            video_contents.video_url,
            link_contents.link_url,
            audio_contents.audio_url,
            title.value AS title,
            description.value AS description,
            content.value AS content,
            long_description.value AS long_description
    FROM contents
    LEFT JOIN text_contents on contents.id = text_contents.content_id
    LEFT JOIN video_contents on contents.id = video_contents.content_id
    LEFT JOIN link_contents on contents.id = link_contents.content_id
    LEFT JOIN audio_contents on contents.id = audio_contents.content_id
    LEFT JOIN translations AS title on title.id = contents.title
    LEFT JOIN translations AS description on description.id = contents.description
    LEFT JOIN translations AS content on content.id = text_contents.content
    LEFT JOIN translations AS long_description on long_description.id = video_contents.long_description
    WHERE status = 1
    AND (text_contents.content_id = contents.id or video_contents.content_id = contents.id or link_contents.content_id = contents.id or audio_contents.content_id = contents.id)
  """

  val SelectOneName = """
    SELECT id, translated_id, version, title, description, content_type, status, created_at, updated_at
    FROM contents
    WHERE title = ?
      AND status = 1
  """

  override val SelectOne = """
    SELECT  contents.id,
            contents.translated_id,
            contents.slug,
            contents.version,
            content_type,
            primary_tag,
            starred,
            contents.status,
            contents.created_at,
            contents.updated_at,
            video_contents.video_type,
            video_contents.video_url,
            link_contents.link_url,
            audio_contents.audio_url,
            tag_lines_contents.position AS tag_line_position,
            title.value AS title,
            description.value AS description,
            content.value AS content,
            long_description.value AS long_description
    FROM contents
    LEFT JOIN text_contents on contents.id = text_contents.content_id
    LEFT JOIN video_contents on contents.id = video_contents.content_id
    LEFT JOIN link_contents on contents.id = link_contents.content_id
    LEFT JOIN audio_contents on contents.id = audio_contents.content_id
    LEFT JOIN tag_lines_contents on contents.id = tag_lines_contents.content_id
    LEFT JOIN translations AS title on title.id = contents.title
    LEFT JOIN translations AS description on description.id = contents.description
    LEFT JOIN translations AS content on content.id = text_contents.content
    LEFT JOIN translations AS long_description on long_description.id = video_contents.long_description
    WHERE contents.id = ?
      AND contents.status = 1
  """

  def SelectByTag(tags: String, language: String): String = {
    var tagArray = tags.split(",")
    var count = tagArray.length
    var whereClause = ""

    for(i <- 0 to (tagArray.length - 1)) {
      whereClause += "'" + tagArray(i) + "'"
      if (i != (tagArray.length - 1)) whereClause += ", "
    }

    """
    WITH id_count as (
      SELECT  contents.id,
              COUNT(contents.id) AS count
      FROM contents
      INNER JOIN tags_contents on tags_contents.content_id = contents.id
      INNER JOIN tags on tags_contents.tag_id = tags.id
      LEFT JOIN text_contents on contents.id = text_contents.content_id
      LEFT JOIN video_contents on contents.id = video_contents.content_id
      LEFT JOIN link_contents on contents.id = link_contents.content_id
      LEFT JOIN audio_contents on contents.id = audio_contents.content_id
      LEFT JOIN translations AS name on name.id = tags.name
      LEFT JOIN translations AS title on title.id = contents.title
      WHERE name.value IN (""" + whereClause + """)
      AND (text_contents.content_id = contents.id or video_contents.content_id = contents.id or link_contents.content_id = contents.id or audio_contents.content_id = contents.id)
      AND title.language_id = (SELECT id FROM languages WHERE name = '""" + language + """')
      GROUP BY contents.id
    )
    SELECT  DISTINCT ON (contents.id)
            contents.id,
            contents.translated_id,
            contents.slug,
            contents.version,
            content_type,
            primary_tag,
            starred,
            contents.status,
            contents.created_at,
            contents.updated_at,
            video_contents.video_type,
            video_contents.video_url,
            link_contents.link_url,
            audio_contents.audio_url,
            title.value AS title,
            description.value AS description,
            content.value AS content,
            long_description.value AS long_description
    FROM contents
    INNER JOIN tags_contents on tags_contents.content_id = contents.id
    INNER JOIN tags on tags_contents.tag_id = tags.id
    LEFT JOIN text_contents on contents.id = text_contents.content_id
    LEFT JOIN video_contents on contents.id = video_contents.content_id
    LEFT JOIN link_contents on contents.id = link_contents.content_id
    LEFT JOIN audio_contents on contents.id = audio_contents.content_id
    LEFT JOIN translations AS title on title.id = contents.title
    LEFT JOIN translations AS description on description.id = contents.description
    LEFT JOIN translations AS content on content.id = text_contents.content
    LEFT JOIN translations AS long_description on long_description.id = video_contents.long_description
    LEFT JOIN translations AS name on name.id = tags.name
    WHERE contents.id IN (SELECT id FROM id_count WHERE count = '""" + count + """')
    """
  }

  val SelectByLanguage = """
    SELECT  contents.id,
            contents.translated_id,
            contents.slug,
            contents.version,
            content_type,
            primary_tag,
            starred,
            contents.status,
            contents.created_at,
            contents.updated_at,
            video_contents.video_type,
            video_contents.video_url,
            link_contents.link_url,
            audio_contents.audio_url,
            title.value AS title,
            description.value AS description,
            content.value AS content,
            long_description.value AS long_description
    FROM contents
    LEFT JOIN text_contents on contents.id = text_contents.content_id
    LEFT JOIN video_contents on contents.id = video_contents.content_id
    LEFT JOIN link_contents on contents.id = link_contents.content_id
    LEFT JOIN audio_contents on contents.id = audio_contents.content_id
    LEFT JOIN translations AS title on title.id = contents.title
    LEFT JOIN translations AS description on description.id = contents.description
    LEFT JOIN translations AS content on content.id = text_contents.content
    LEFT JOIN translations AS long_description on long_description.id = video_contents.long_description
    WHERE (text_contents.content_id = contents.id or video_contents.content_id = contents.id or link_contents.content_id = contents.id or audio_contents.content_id = contents.id)
    AND title.language_id = (SELECT id FROM languages WHERE name = ?)
    """

  val SelectUntagged = """
    SELECT  contents.id,
            contents.translated_id,
            contents.slug,
            contents.version,
            content_type,
            primary_tag,
            starred,
            contents.status,
            contents.created_at,
            contents.updated_at,
            video_contents.video_type,
            video_contents.video_url,
            link_contents.link_url,
            audio_contents.audio_url,
            title.value AS title,
            description.value AS description,
            content.value AS content,
            long_description.value AS long_description
    FROM contents
    LEFT JOIN text_contents on contents.id = text_contents.content_id
    LEFT JOIN video_contents on contents.id = video_contents.content_id
    LEFT JOIN link_contents on contents.id = link_contents.content_id
    LEFT JOIN audio_contents on contents.id = audio_contents.content_id
    LEFT JOIN translations AS title on title.id = contents.title
    LEFT JOIN translations AS description on description.id = contents.description
    LEFT JOIN translations AS content on content.id = text_contents.content
    LEFT JOIN translations AS long_description on long_description.id = video_contents.long_description
    WHERE (text_contents.content_id = contents.id or video_contents.content_id = contents.id or link_contents.content_id = contents.id or audio_contents.content_id = contents.id)
    AND title.language_id = (SELECT id FROM languages WHERE name = ?)
    AND contents.id NOT IN (SELECT content_id FROM tags_contents)
  """

  val SelectById = """
    SELECT  contents.id,
            contents.translated_id,
            contents.slug,
            contents.version,
            contents.content_type,
            contents.primary_tag,
            contents.starred,
            contents.status,
            contents.created_at,
            contents.updated_at,
            video_contents.video_type,
            video_contents.video_url,
            link_contents.link_url,
            audio_contents.audio_url,
            tag_lines_contents.position AS tag_line_position,
            featured_contents.position AS featured_position,
            site_word_contents.position AS site_word_position,
            select_featured_contents.position AS select_featured_position,
            header_items_contents.position AS header_item_position,
            footer_items_contents.position AS footer_item_position,
            featured_contents.icon_type,
            select_featured_contents.icon_type AS select_featured_icon_type,
            title.value AS title,
            description.value AS description,
            content.value AS content,
            long_description.value AS long_description,
            header_content.value AS header_content,
            footer_content.value AS footer_content,
            featured_content.value AS featured_content,
            site_word_contents.href_id as site_word_href_id,
            select_featured_contents.href_slug as select_featured_href_slug,
            site_contents.site_logo,
            site_contents.content as site_content,
            site_contents.site_url,
            slide_contents.position as slide_position,
            slide_contents.slide_background,
            slide_contents.content as slide_content,
            slide_contents.slide_url,
            slide_contents.header_colour
    FROM contents
    LEFT JOIN text_contents on contents.id = text_contents.content_id
    LEFT JOIN video_contents on contents.id = video_contents.content_id
    LEFT JOIN link_contents on contents.id = link_contents.content_id
    LEFT JOIN audio_contents on contents.id = audio_contents.content_id
    LEFT JOIN tag_lines_contents on contents.id = tag_lines_contents.content_id
    LEFT JOIN header_items_contents on contents.id = header_items_contents.content_id
    LEFT JOIN footer_items_contents on contents.id = footer_items_contents.content_id
    LEFT JOIN featured_contents on contents.id = featured_contents.content_id
    LEFT JOIN select_featured_contents on contents.id = select_featured_contents.content_id
    LEFT JOIN site_word_contents on contents.id = site_word_contents.content_id
    LEFT JOIN site_contents on contents.id = site_contents.content_id
    LEFT JOIN slide_contents on contents.id = slide_contents.content_id
    LEFT JOIN translations AS title on title.id = contents.title
    LEFT JOIN translations AS description on description.id = contents.description
    LEFT JOIN translations AS content on content.id = text_contents.content
    LEFT JOIN translations AS long_description on long_description.id = video_contents.long_description
    LEFT JOIN translations AS header_content on header_content.id = header_items_contents.content
    LEFT JOIN translations AS footer_content on footer_content.id = footer_items_contents.content
    LEFT JOIN translations AS featured_content on featured_content.id = featured_contents.content
    WHERE contents.id = ?
      AND contents.status = 1
  """

  val SelectByTitle = """
    SELECT  contents.id,
            contents.translated_id,
            contents.slug,
            contents.version,
            contents.content_type,
            contents.primary_tag,
            contents.starred,
            contents.status,
            contents.created_at,
            contents.updated_at,
            video_contents.video_type,
            video_contents.video_url,
            link_contents.link_url,
            audio_contents.audio_url,
            title.value AS title,
            description.value AS description,
            content.value AS content,
            long_description.value AS long_description,
            featured_content.value AS featured_content,
            featured_contents.icon_type,
            featured_contents.position AS featured_position
    FROM contents
    LEFT JOIN text_contents on contents.id = text_contents.content_id
    LEFT JOIN video_contents on contents.id = video_contents.content_id
    LEFT JOIN link_contents on contents.id = link_contents.content_id
    LEFT JOIN audio_contents on contents.id = audio_contents.content_id
    LEFT JOIN featured_contents on contents.id = featured_contents.content_id
    LEFT JOIN translations AS title on title.id = contents.title
    LEFT JOIN translations AS description on description.id = contents.description
    LEFT JOIN translations AS content on content.id = text_contents.content
    LEFT JOIN translations AS featured_content on featured_content.id = featured_contents.content
    LEFT JOIN translations AS long_description on long_description.id = video_contents.long_description
    WHERE title.value = ?
      AND contents.status = 1
    LIMIT 1
  """

  val SelectBySlug = """
    SELECT  contents.id,
            contents.translated_id,
            contents.slug,
            contents.version,
            contents.content_type,
            contents.primary_tag,
            contents.starred,
            contents.status,
            contents.created_at,
            contents.updated_at,
            video_contents.video_type,
            video_contents.video_url,
            link_contents.link_url,
            audio_contents.audio_url,
            title.value AS title,
            description.value AS description,
            content.value AS content,
            long_description.value AS long_description,
            featured_content.value AS featured_content,
            featured_contents.icon_type,
            featured_contents.position AS featured_position,
            select_featured_contents.position AS select_featured_position,
            select_featured_contents.href_slug AS select_featured_href_slug,
            select_featured_contents.icon_type AS select_featured_icon_type
    FROM contents
    LEFT JOIN text_contents on contents.id = text_contents.content_id
    LEFT JOIN video_contents on contents.id = video_contents.content_id
    LEFT JOIN link_contents on contents.id = link_contents.content_id
    LEFT JOIN audio_contents on contents.id = audio_contents.content_id
    LEFT JOIN featured_contents on contents.id = featured_contents.content_id
    LEFT JOIN select_featured_contents on contents.id = select_featured_contents.content_id
    LEFT JOIN translations AS title on title.id = contents.title
    LEFT JOIN translations AS description on description.id = contents.description
    LEFT JOIN translations AS content on content.id = text_contents.content
    LEFT JOIN translations AS featured_content on featured_content.id = featured_contents.content
    LEFT JOIN translations AS long_description on long_description.id = video_contents.long_description
    WHERE slug = ?
      AND contents.status = 1
    LIMIT 1
  """

  val SetPrimaryTag = """
    UPDATE contents
    SET primary_tag = ?
    WHERE id = ?
    RETURNING version
  """

  val Delete = """
    WITH content AS (
      SELECT id, translated_id
      FROM contents
      WHERE contents.id = ?
    ), tag_content AS (
      SELECT tag_id
      FROM tags_contents
      WHERE content_id = (SELECT id FROM content)
    ), delete_content AS (
      DELETE FROM contents
      WHERE contents.id = (SELECT id FROM content)
        OR contents.id = (SELECT translated_id FROM content)
      RETURNING *
    ), tag AS (
      UPDATE tags
      SET count = count - 1
      WHERE tags.id IN (SELECT tag_content.tag_id FROM tag_content)
    )
    SELECT count(*) FROM delete_content
  """

  def update(id: Array[Byte], version: Long, fields: Map[String, Any]) {
    val UpdatePrefix = "UPDATE contents SET "
    val UpdatePostfix = " WHERE id = ? AND version = ? AND status = 1 RETURNING version"

    val query = fields.map { case (key, value) =>
      key + " = ?"
    }.mkString(",")

    val values = fields.map({ case (key, value) =>
      value
    }).toArray:+(id):+(version)

    val UpdateQuery = UpdatePrefix + query + UpdatePostfix
    pool.sendPreparedStatement(UpdateQuery, values).map {
      queryResult => queryResult.rows.get(0)("version").asInstanceOf[Long]
    }
  }

  /**
   * Find a single entry by ID.
   *
   * @param id the 128-bit UUID, as a byte array, to search for.
   * @return an optional RowData object containing the results
   */
  def find(id: Array[Byte]): Future[IndexedSeq[RowData]] = {
    pool.sendPreparedStatement(SelectById, Array(id)).map(_.rows.get)
  }

  def findByTitle(title: String): Future[IndexedSeq[RowData]] = {
    pool.sendPreparedStatement(SelectByTitle, Array(title)).map(_.rows.get)
  }

  def findBySlug(slug: String): Future[IndexedSeq[RowData]] = {
    pool.sendPreparedStatement(SelectBySlug, Array(slug)).map(_.rows.get)
  }

  /**
   * Find a single Content Entity by title.
   *
   * @param title the content title to search by
   * @return the future optional RowData returned from the database
   */
  def findByName(title: String): Future[Option[RowData]] = {
    for {
      queryResult <- pool.sendPreparedStatement(SelectOneName, Array(title.replace("'", "''")))
      maybeRow <- Future { queryResult.rows match {
        case Some(rows) => {
          rows.headOption match {
            case Some(rowData) => Some(rowData)
            case None => None
          }
        }
        case None => None
      }}
    } yield maybeRow
  }

  def findByTags(tags: String, language: String): Future[ResultSet] = {
    pool.sendQuery(SelectByTag(tags.replace("'", "''"), language)).map { queryResult =>
      queryResult.rows.get
    }
  }

  def findByLanguage(language: String): Future[ResultSet] = {
    pool.sendPreparedStatement(SelectByLanguage, Array(language)).map { queryResult =>
      queryResult.rows.get
    }
  }

  def findUntagged(language: String): Future[ResultSet] = {
    pool.sendPreparedStatement(SelectUntagged, Array(language)).map { queryResult =>
      queryResult.rows.get
    }
  }

  def findLanguage(id: Array[Byte]): Future[IndexedSeq[RowData]] = {
    pool.sendPreparedStatement(SelectLanguage, Array(id)).map(_.rows.get)
  }

  def findSlug(id: Array[Byte]): Future[IndexedSeq[RowData]] = {
    pool.sendPreparedStatement(SelectSlug, Array(id)).map(_.rows.get)
  }

  /**
   * Save a Content row.
   *
   * @return id of the saved/new content.
   */
  def update(id: Array[Byte],
             version: Long,
             title: Array[Byte],
             description: Array[Byte]): Future[Option[Long]] = {
    pool.sendPreparedStatement(Update, Array(version, title, description, new DateTime, id)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("version").asInstanceOf[Long])
            case None => {
              Logger.debug(queryResult.statusMessage)
              None
            }
          }
          case None => {
            Logger.debug(queryResult.statusMessage)
            None
          }
        }
      }
    }
  }

  /**
   * Inserts a new content.
   */
  def insert(id: Array[Byte],
             translated_id: Array[Byte],
             title: Array[Byte],
             description: Array[Byte],
             contentType: String): Future[Option[Long]] = {
    pool.sendPreparedStatement(Insert, Array(id, translated_id, title, description, contentType, new DateTime, new DateTime)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("version").asInstanceOf[Long])
            case None => {
              Logger.debug(queryResult.statusMessage)
              None
            }
          }
          case None => {
            Logger.debug(queryResult.statusMessage)
            None
          }
        }
      }
    }
  }

  def setPrimaryTag(id: Array[Byte], primaryTag: String): Future[Boolean] = {
    for {
      queryResult <- pool.sendPreparedStatement(SetPrimaryTag, Array(primaryTag, id))
    } yield { queryResult.rowsAffected > 0 }
  }

  /**
   *
   * @param id
   * @return
   */
  def delete(id: Array[Byte]): Future[Boolean] = {
    for {
      queryResult <- pool.sendPreparedStatement(Delete, Array(id))
    } yield { queryResult.rowsAffected > 0 }
  }
}
