package services.datasource

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports._
import webcrank.password._
import scala.concurrent.Future
import org.joda.time.LocalDate
import play.api.Logger

import helpers.UUID

object FeaturedTDG extends TableDataGateway {

  val Insert = """
    WITH content AS (
      INSERT INTO contents (id, translated_id, version, title, slug, description, content_type, primary_tag, starred, status, created_at, updated_at)
      VALUES (?, ?, 1, ?, ?, ?, ?, ?, ?, 1, ?, ?)
      RETURNING id, translated_id, version, title, description
    ), featured AS (
      INSERT INTO featured_contents (content_id, position, icon_type, content)
      SELECT (SELECT id FROM content), ? AS position, ? AS icon_type, ? AS content
      RETURNING content_id, position, icon_type, content
    ), translations as (
      INSERT INTO translations (id, language_id, value)
      SELECT title, (SELECT id FROM languages WHERE name = ?), ? AS value
      FROM content
      UNION ALL
      SELECT description, (SELECT id FROM languages WHERE name = ?), ? AS value
      FROM content
      UNION ALL
      SELECT content, (SELECT id FROM languages WHERE name = ?), ? AS value
      FROM featured
    ), log as (
      INSERT INTO logs (id, content_id, created_by, created_at)
      VALUES (?, ?, ?, now())
    )
    SELECT *
    FROM content
  """

  val Update = """
    WITH content AS (
      UPDATE contents
      SET version = version + 1, primary_tag = ?, starred = ?, slug = ?, updated_at = now()
      WHERE status = 1
        AND id = ?
      RETURNING id, version, title, description
    ), featured AS (
      UPDATE featured_contents
      SET position = ?, icon_type = ?
      WHERE content_id = (SELECT id FROM content)
      RETURNING content
    ), translations as (
      UPDATE translations
      SET value =
        CASE
          WHEN id = (SELECT title FROM content) THEN ?
          WHEN id = (SELECT description FROM content) THEN ?
          WHEN id = (SELECT content FROM featured) THEN ?
          ELSE value
        END
    ), log as (
      UPDATE logs
      SET updated_by = ?, updated_at = now()
      WHERE content_id = ?
    )
    SELECT *
    FROM content
  """

  val Delete = """
    UPDATE contents
    SET status = 0
    WHERE id = ?
      AND version = ?
  """

  override val SelectAll = """
    SELECT  contents.id,
            contents.translated_id,
            contents.version,
            content_type,
            contents.status,
            contents.created_at,
            contents.updated_at,
            contents.starred,
            featured_contents.position,
            featured_contents.icon_type,
            title.value AS title,
            description.value AS description,
            featured_content.value AS featured_content
    FROM contents
    LEFT JOIN featured_contents on contents.id = featured_contents.content_id
    LEFT JOIN translations AS title on title.id = contents.title
    LEFT JOIN translations AS description on description.id = contents.description
    LEFT JOIN translations AS featured_content on featured_content.id = featured_contents.content
    WHERE featured_contents.content_id = contents.id
      AND title.language_id = (SELECT id FROM languages WHERE name = ?)
  """

  val SelectByLanguage = """
    SELECT  contents.id,
            contents.translated_id,
            contents.slug,
            contents.version,
            contents.content_type,
            contents.primary_tag,
            contents.starred,
            contents.status,
            contents.created_at,
            contents.updated_at,
            featured_contents.position AS featured_position,
            featured_contents.icon_type,
            title.value AS title,
            description.value AS description,
            featured_content.value AS featured_content
    FROM contents
    LEFT JOIN featured_contents on contents.id = featured_contents.content_id
    LEFT JOIN translations AS title on title.id = contents.title
    LEFT JOIN translations AS description on description.id = contents.description
    LEFT JOIN translations AS featured_content on featured_content.id = featured_contents.content
    WHERE featured_contents.content_id = contents.id
    AND title.language_id = (SELECT id FROM languages WHERE name = ?)
  """

  /**
   * Update a Featured row.
   *
   * @return id of the saved/new audio.
   */
  def update(id: Array[Byte],
             title: String,
             slug: String,
             description: String,
             position: Integer,
             icon_type: String,
             content: String,
             primary_tag: String,
             starred: Integer,
             user_id: Array[Byte]): Future[Option[Long]] = {
    pool.sendPreparedStatement(Update, Array(
      primary_tag, starred, slug, id, position, icon_type, title, description, content, user_id, id)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("version").asInstanceOf[Long])
            case None => {
              Logger.debug(queryResult.statusMessage)
              None
            }
          }
          case None => {
            Logger.debug(queryResult.statusMessage)
            None
          }
        }
      }
    }
  }

  /**
   * Inserts a new text.
   */
  def insert(id: Array[Byte],
             translated_id: Array[Byte],
             title: String,
             slug: String,
             description: String,
             position: Integer,
             icon_type: String,
             content: String,
             content_type: String,
             primary_tag: String,
             starred: Integer,
             language: String,
             user_id: Array[Byte]): Future[Option[Long]] = {
    pool.sendPreparedStatement(Insert, Array(
      id, translated_id, UUID.random, slug, UUID.random, content_type, primary_tag, starred, new DateTime, new DateTime,
      position, icon_type, UUID.random, language, title, language, description, language, content, UUID.random, id, user_id)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("version").asInstanceOf[Long])
            case None => None
          }
          case None => None
        }
      }
    }
  }

  def findByLanguage(language: String): Future[ResultSet] = {
    pool.sendPreparedStatement(SelectByLanguage, Array(language)).map { queryResult =>
      queryResult.rows.get
    }
  }

  /**
   * Delete a Text from the database.
   *
   * @param id the uuid of the text to delete
   * @return a boolean indicating whether the operation was successful
   */
  def delete(id: Array[Byte]): Future[Boolean] = {
    pool.sendPreparedStatement(Delete, Array(id)).map(_.rowsAffected > 0)
  }
}
