package services.datasource

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports._
import webcrank.password._
import scala.concurrent.Future
import org.joda.time.LocalDate
import play.api.Logger

import helpers.UUID

object FooterItemTDG extends TableDataGateway {

  val Insert = """
    WITH content AS (
      INSERT INTO contents (id, translated_id, version, title, slug, description, content_type, status, created_at, updated_at)
      VALUES (?, ?, 1, ?, ?, ?, ?, 1, ?, ?)
      RETURNING id, translated_id, version, title, description
    ), footer_item AS (
      INSERT INTO footer_items_contents (content_id, position, content)
      SELECT (SELECT id FROM content), ? AS position, ? AS content
      RETURNING content_id, position, content
    ), translations as (
      INSERT INTO translations (id, language_id, value)
      SELECT title, (SELECT id FROM languages WHERE name = ?), ? AS value
      FROM content
      UNION ALL
      SELECT description, (SELECT id FROM languages WHERE name = ?), ? AS value
      FROM content
      UNION ALL
      SELECT content, (SELECT id FROM languages WHERE name = ?), ? AS value
      FROM footer_item
    ), log as (
      INSERT INTO logs (id, content_id, created_by, created_at)
      VALUES (?, ?, ?, now())
    )
    SELECT *
    FROM content
  """

  val Update = """
    WITH content AS (
      UPDATE contents
      SET version = version + 1, updated_at = now()
      WHERE status = 1
        AND id = ?
      RETURNING id, version, title, description
    ), footerItem AS (
      UPDATE footer_items_contents
      SET position = ?
      WHERE content_id = (SELECT id FROM content)
      RETURNING content
    ), translations as (
      UPDATE translations
      SET value =
        CASE
          WHEN id = (SELECT title FROM content) THEN ?
          WHEN id = (SELECT description FROM content) THEN ?
          WHEN id = (SELECT content FROM footerItem) THEN ?
          ELSE value
        END
    ), log as (
      UPDATE logs
      SET updated_by = ?, updated_at = now()
      WHERE content_id = ?
    )
    SELECT *
    FROM content
  """

  val Delete = """
    UPDATE contents
    SET status = 0
    WHERE id = ?
      AND version = ?
  """

  override val SelectAll = """
    SELECT  contents.id,
            contents.translated_id,
            contents.version,
            content_type,
            contents.status,
            contents.created_at,
            contents.updated_at,
            footer_items_contents.position,
            footer_items_contents.content AS content,
            title.value AS title,
            description.value AS description
    FROM contents
    LEFT JOIN footer_items_contents on contents.id = footer_items_contents.content_id
    LEFT JOIN translations AS title on title.id = contents.title
    LEFT JOIN translations AS description on description.id = contents.description
    WHERE status = 1
    AND footer_items_contents.content_id = contents.id
    AND title.language_id = (SELECT id FROM languages WHERE name = 'english')
  """

  val SelectByLanguage =  """
    SELECT  contents.id,
            contents.translated_id,
            contents.version,
            content_type,
            primary_tag,
            starred,
            contents.status,
            contents.created_at,
            contents.updated_at,
            footer_items_contents.position AS footer_item_position,
            title.value AS title,
            description.value AS description,
            footer_content.value AS footer_content
    FROM contents
    LEFT JOIN footer_items_contents on contents.id = footer_items_contents.content_id
    LEFT JOIN translations AS title on title.id = contents.title
    LEFT JOIN translations AS description on description.id = contents.description
    LEFT JOIN translations AS footer_content on footer_content.id = footer_items_contents.content
    WHERE footer_items_contents.content_id = contents.id
    AND title.language_id = (SELECT id FROM languages WHERE name = ?)
  """

  override val SelectOne = """
    SELECT *
    FROM contents, footer_items_contents
    WHERE status = 1
      AND id = ?
      AND version = ?
      AND id = content_id
  """

  val SelectById = """
    SELECT *
    FROM contents, footer_items_contents
    WHERE status = 1
      AND id = footer_items_contents.content_id
      AND content_group_id = ?
  """

  /**
   * Find a single entry by ID.
   *
   * @param id the 128-bit UUID, as a byte array, to search for.
   * @return an optional RowData object containing the results
   */
  def find(id: Array[Byte]): Future[IndexedSeq[RowData]] = {
    pool.sendPreparedStatement(SelectById, Array(id)).map(_.rows.get)
  }

  def findByLanguage(language: String): Future[ResultSet] = {
    pool.sendPreparedStatement(SelectByLanguage, Array(language)).map { queryResult =>
      queryResult.rows.get
    }
  }

  /**
   * Update a FooterItem row.
   *
   * @return id of the saved/new audio.
   */
  def update(id: Array[Byte],
             title: String,
             slug: String,
             description: String,
             position: Integer,
             content: String,
             user_id: Array[Byte]): Future[Option[Long]] = {
    pool.sendPreparedStatement(Update, Array(
      id, position, title, description, content, user_id, id)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("version").asInstanceOf[Long])
            case None => None
          }
          case None => None
        }
      }
    }
  }

  /**
   * Inserts a new text.
   */
  def insert(id: Array[Byte],
             translated_id: Array[Byte],
             title: String,
             slug: String,
             description: String,
             position: Integer,
             content: String,
             content_type: String,
             language: String,
             user_id: Array[Byte]): Future[Option[Long]] = {
    pool.sendPreparedStatement(Insert, Array(
      id, translated_id, UUID.random, slug, UUID.random, content_type, new DateTime, new DateTime,
      position, UUID.random, language, title, language, description, language, content, UUID.random, id, user_id)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("version").asInstanceOf[Long])
            case None => None
          }
          case None => None
        }
      }
    }
  }

  /**
   * Delete a Text from the database.
   *
   * @param id the uuid of the text to delete
   * @return a boolean indicating whether the operation was successful
   */
  def delete(id: Array[Byte]): Future[Boolean] = {
    pool.sendPreparedStatement(Delete, Array(id)).map(_.rowsAffected > 0)
  }
}
