package services.datasource

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports._

import scala.concurrent.Future

import helpers.UUID

import play.api.Logger

object HomeBlockTDG extends TableDataGateway {

  override val SelectOne = """
    SELECT  id,
            version,
            language_id,
            (SELECT name FROM languages WHERE name = ?) AS language,
            title,
            description,
            content
    FROM home_blocks
    WHERE id = ?
      AND language_id = (SELECT id FROM languages WHERE name = ?)
    LIMIT 1
  """

  val SelectByLanguage = """
    SELECT  id,
            version,
            language_id,
            (SELECT name FROM languages WHERE name = ?) AS language,
            title,
            description,
            content
    FROM home_blocks
    WHERE language_id = (SELECT id FROM languages WHERE name = ?)
    LIMIT 1
  """

  val Insert = """
    INSERT INTO home_blocks (id, version, language_id, title, description, content)
    VALUES (?, ?, (SELECT id FROM languages WHERE name = ?), ?, ?, ?)
    RETURNING version
  """

  val Update = """
    UPDATE home_blocks
    SET version = ?, title = ?, description = ?, content = ?
    WHERE language_id = (SELECT id FROM languages WHERE name = ?)
    RETURNING version
  """

  val Delete = """
    DELETE FROM home_blocks
    WHERE id = ?
  """

  def findById(id: Array[Byte], language: String): Future[Option[RowData]] = {
    for {
      queryResult <- pool.sendPreparedStatement(SelectOne, Array(language, id, language))
      maybeRow <- Future { queryResult.rows match {
        case Some(rows) => {
          if (queryResult.rowsAffected > 0) {
            Some(rows.apply(0))
          }
          else {
            None
          }
        }
        case None => None
      }}
    } yield maybeRow
  }

  def findByLanguage(language: String): Future[Option[RowData]] = {
    for {
      queryResult <- pool.sendPreparedStatement(SelectByLanguage, Array(language, language))
      maybeRow <- Future { queryResult.rows match {
        case Some(rows) => {
          if (queryResult.rowsAffected > 0) {
            Some(rows.apply(0))
          }
          else {
            None
          }
        }
        case None => None
      }}
    } yield maybeRow
  }

  /**
   * Inserts a new HomeBlock.
   *
   * @return the version as a future option long
   */
  def insert( id: Array[Byte],
              version: Long,
              language: String,
              title: String,
              description: String,
              content: String): Future[Option[Long]] = {
    pool.sendPreparedStatement(Insert, Array(id, version, language, title, description, content)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("version").asInstanceOf[Long])
            case None => None
          }
          case None => None
        }
      }
    }
  }

  /**
   * Updates an existing HomeBlock.
   *
   * @return the version as a future option long
   */
  def update( id: Array[Byte],
              version: Long,
              language: String,
              title: String,
              description: String,
              content: String): Future[Option[Long]] = {
    pool.sendPreparedStatement(Update, Array(version + 1, title, description, content, language)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("version").asInstanceOf[Long])
            case None => None
          }
          case None => None
        }
      }
    }
  }

  /**
   * Delete a HomeBlock from the database.
   *
   * @param id the id of the HomeBlock to delete
   * @return a boolean indicating whether the operation was successful
   */
  def delete(id: Array[Byte]): Future[Boolean] = {
    pool.sendPreparedStatement(Delete, Array(id)).map(_.rowsAffected > 0)
  }

}
