package services.datasource

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports._
import webcrank.password._
import scala.concurrent.Future
import org.joda.time.LocalDate
import play.api.Logger

import helpers.UUID

object LinkTDG extends TableDataGateway {

  val Insert = """
    WITH content AS (
      INSERT INTO contents (id, translated_id, version, title, slug, description, content_type, primary_tag, starred, status, created_at, updated_at)
      VALUES (?, ?, 1, ?, ?, ?, ?, ?, ?, 1, ?, ?)
      RETURNING id, version, title, description
    ), link AS (
      INSERT INTO link_contents (content_id, link_url)
      SELECT id, ? AS link_url
      FROM content
      RETURNING link_url
    ), translations as (
      INSERT INTO translations (id, language_id, value)
      SELECT title, (SELECT id FROM languages WHERE name = ?), ? AS value
      FROM content
      UNION ALL
      SELECT description, (SELECT id FROM languages WHERE name = ?), ? AS value
      FROM content
    ), log as (
      INSERT INTO logs (id, content_id, created_by, created_at)
      VALUES (?, ?, ?, now())
    )
    SELECT *
    FROM content
  """

  val Update = """
    WITH content AS (
      UPDATE contents
      SET version = version + 1, primary_tag = ?, starred = ?, slug = ?, updated_at = now()
      WHERE status = 1
        AND id = ?
      RETURNING id, version, title, description
    ), link AS (
      UPDATE link_contents
      SET link_url = ?
      WHERE content_id = (SELECT id FROM content)
    ), translations as (
      UPDATE translations
      SET value =
        CASE
          WHEN id = (SELECT title FROM content) THEN ?
          WHEN id = (SELECT description FROM content) THEN ?
          ELSE value
        END
    ), log as (
      UPDATE logs
      SET updated_by = ?, updated_at = now()
      WHERE content_id = ?
    )
    SELECT *
    FROM content
  """

  val Delete = """
    UPDATE contents
    SET status = 0
    WHERE id = ?
      AND version = ?
  """

  override val SelectAll = """
    SELECT *
    FROM link_contents
  """

  /**
   * Update a Link row.
   *
   * @return id of the saved/new link.
   */
  def update(id: Array[Byte],
             title: String,
             slug: String,
             description: String,
             link_url: String,
             primary_tag: String,
             starred: Integer,
             user_id: Array[Byte]): Future[Option[Long]] = {
    pool.sendPreparedStatement(Update, Array(
      primary_tag, starred, slug, id, link_url, title, description, user_id, id)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("version").asInstanceOf[Long])
            case None => None
          }
          case None => None
        }
      }
    }
  }

  /**
   * Inserts a new link.
   */
  def insert(id: Array[Byte],
             translated_id: Array[Byte],
             title: String,
             slug: String,
             description: String,
             content_type: String,
             primary_tag: String,
             starred: Integer,
             link_url: String,
             language: String,
             user_id: Array[Byte]): Future[Option[Long]] = {
    pool.sendPreparedStatement(Insert, Array(
      id, translated_id, UUID.random, slug, UUID.random, content_type, primary_tag, starred, new DateTime, new DateTime,
      link_url, language, title, language, description, UUID.random, id, user_id)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("version").asInstanceOf[Long])
            case None => None
          }
          case None => None
        }
      }
    }
  }

  /**
   * Delete a Link from the database.
   *
   * @param id the uuid of the link to delete
   * @return a boolean indicating whether the operation was successful
   */
  def delete(id: Array[Byte]): Future[Boolean] = {
    pool.sendPreparedStatement(Delete, Array(id)).map(_.rowsAffected > 0)
  }
}
