package services.datasource

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports._

import scala.concurrent.Future

import helpers.UUID

import play.api.Logger

object MenuItemTDG extends TableDataGateway {

  override val SelectOne = """
    SELECT  id,
            version,
            parent_id,
            tagline_id,
            content_id,
            language_id,
            (SELECT name FROM languages WHERE name = ?) AS language,
            position,
            title,
            description,
            long_description,
            published
    FROM menu_items
    WHERE id = ?
      AND language_id = (SELECT id FROM languages WHERE name = ?)
  """

  val SelectAllParents = """
    SELECT  id,
            version,
            parent_id,
            tagline_id,
            content_id,
            language_id,
            (SELECT name FROM languages WHERE name = ?) AS language,
            position,
            title,
            description,
            long_description,
            published
    FROM menu_items
    WHERE parent_id IS NULL
      AND language_id = (SELECT id FROM languages WHERE name = ?)
  """

  val SelectByParentId = """
    SELECT  id,
            version,
            parent_id,
            tagline_id,
            content_id,
            language_id,
            (SELECT name FROM languages WHERE name = ?) AS language,
            position,
            title,
            description,
            long_description,
            published
    FROM menu_items
    WHERE parent_id = ?
      AND language_id = (SELECT id FROM languages WHERE name = ?)
  """

  val SelectLargestPosition = """
    SELECT position
    FROM menu_items
    ORDER BY position DESC LIMIT 1
  """

  val SetPosition = """
    UPDATE menu_items
    SET position = ?
    WHERE id = ?
  """

  val SelectLargestPositionByParent = """
    SELECT position
    FROM menu_items
    WHERE parent_id = ?
    ORDER BY position DESC LIMIT 1
  """

  val Insert = """
    INSERT INTO menu_items (id,
                            version,
                            parent_id,
                            tagline_id,
                            content_id,
                            language_id,
                            position,
                            title,
                            description,
                            long_description,
                            published)
    VALUES (?, ?, ?, ?, ?, (SELECT id FROM languages WHERE name = ?), ?, ?, ?, ?, ?)
    RETURNING version
  """

  val Update = """
    UPDATE menu_items
    SET version = ?,
        parent_id = ?,
        tagline_id = ?,
        content_id = ?,
        position = ?,
        title = ?,
        description = ?,
        long_description = ?,
        published = ?
    WHERE id = ?
      AND language_id = (SELECT id FROM languages WHERE name = ?)
    RETURNING version
  """

  val Delete = """
    WITH update_parent_ids AS (
      UPDATE menu_items
      SET parent_id = NULL
      WHERE parent_id = ?
    )
    DELETE FROM menu_items
    WHERE id = ?
  """

  val DeleteCascade = """
    DELETE FROM menu_items
    WHERE parent_id = ?
    OR id = ?
  """

  def findById(id: Array[Byte], language: String): Future[Option[RowData]] = {
    for {
      queryResult <- pool.sendPreparedStatement(SelectOne, Array(language, id, language))
      maybeRow <- Future { queryResult.rows match {
        case Some(rows) => {
          if (queryResult.rowsAffected > 0) {
            Some(rows.apply(0))
          }
          else {
            None
          }
        }
        case None => None
      }}
    } yield maybeRow
  }

  /**
   * Select all the MenuItems with no parents.
   *
   * @return a future indexed sequence of RowData
   */
  def findAllParents(language: String): Future[IndexedSeq[RowData]] = {
    pool.sendPreparedStatement(SelectAllParents, Array(language, language)).map { queryResult =>
      queryResult.rows.get
    }
  }

  /**
   * Select MenuItems by parent id.
   *
   * @return a future indexed sequence of RowData
   */
  def findByParentId(parentId: Array[Byte], language: String): Future[IndexedSeq[RowData]] = {
    pool.sendPreparedStatement(SelectByParentId, Array(language, parentId, language)).map { queryResult =>
      queryResult.rows.get
    }
  }

  def getLargestPosition: Future[Option[Int]] = {
    pool.sendQuery(SelectLargestPosition).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("position").asInstanceOf[Int])
            case None => None
          }
          case None => None
        }
      }
    }
  }

  def getLargestPosition(parentId: Array[Byte]): Future[Option[Int]] = {
    pool.sendPreparedStatement(SelectLargestPositionByParent, Array(parentId)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("position").asInstanceOf[Int])
            case None => None
          }
          case None => None
        }
      }
    }
  }

  def setPosition(id: Array[Byte], position: Int): Future[Boolean] = {
    pool.sendPreparedStatement(SetPosition, Array(position, id)).map(_.rowsAffected > 0)
  }

  /**
   * Inserts a new MenuItem.
   *
   * @return the version as a future option long
   */
  def insert( id: Array[Byte],
              version: Long,
              parentId: Option[String],
              tagLineId: Option[String],
              contentId: Option[String],
              language: String,
              position: Int,
              title: String,
              description: String,
              longDescription: String,
              published: Int): Future[Option[Long]] = {
    val maybeParentId = parentId match {
      case Some(id) => UUID(id).bytes
      case None => null
    }
    val maybeTagLineId = tagLineId match {
      case Some(id) => UUID(id).bytes
      case None => null
    }
    val maybeContentId = contentId match {
      case Some(id) => UUID(id).bytes
      case None => null
    }

    pool.sendPreparedStatement(Insert, Array(id, version, maybeParentId, maybeTagLineId, maybeContentId, language, position, title, description, longDescription, published)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("version").asInstanceOf[Long])
            case None => None
          }
          case None => None
        }
      }
    }
  }

  /**
   * Updates an existing MenuItem.
   *
   * @return the version as a future option long
   */
  def update( id: Array[Byte],
              version: Long,
              parentId: Option[String],
              tagLineId: Option[String],
              contentId: Option[String],
              language: String,
              position: Int,
              title: String,
              description: String,
              longDescription: String,
              published: Int): Future[Option[Long]] = {
    val maybeParentId = parentId match {
      case Some(id) => if (id.length == 0) null else UUID(id).bytes
      case None => null
    }
    val maybeTagLineId = tagLineId match {
      case Some(id) => if (id.length == 0) null else UUID(id).bytes
      case None => null
    }
    val maybeContentId = contentId match {
      case Some(id) => if (id.length == 0) null else UUID(id).bytes
      case None => null
    }

    pool.sendPreparedStatement(Update, Array(version + 1, maybeParentId, maybeTagLineId, maybeContentId, position, title, description, longDescription, published, id, language)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("version").asInstanceOf[Long])
            case None => None
          }
          case None => None
        }
      }
    }
  }

  /**
   * Delete a MenuItem from the database.
   *
   * @param id the id of the MenuItem to delete
   * @return a boolean indicating whether the operation was successful
   */
  def delete(id: Array[Byte]): Future[Boolean] = {
    pool.sendPreparedStatement(Delete, Array(id, id)).map(_.rowsAffected > 0)
  }

  def deleteCascade(id: Array[Byte]): Future[Boolean] = {
    pool.sendPreparedStatement(DeleteCascade, Array(id, id)).map(_.rowsAffected > 0)
  }

}
