package services.datasource

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports._
import webcrank.password._
import scala.concurrent.Future
import org.joda.time.LocalDate
import play.api.Logger

import helpers.UUID

object SelectFeaturedTDG extends TableDataGateway {

  val Insert = """
    WITH content AS (
      INSERT INTO contents (id, translated_id, version, title, slug, description, content_type, primary_tag, starred, status, created_at, updated_at)
      VALUES (?, ?, 1, ?, ?, ?, ?, ?, ?, 1, ?, ?)
      RETURNING id, translated_id, version, title, description
    ), selectFeatured AS (
      INSERT INTO select_featured_contents (content_id, href_slug, position, icon_type)
      SELECT (SELECT id FROM content), ? AS href_slug, ? AS position, ? AS icon_type
      RETURNING content_id
    ), translations as (
      INSERT INTO translations (id, language_id, value)
      SELECT title, (SELECT id FROM languages WHERE name = ?), ? AS value
      FROM content
      UNION ALL
      SELECT description, (SELECT id FROM languages WHERE name = ?), ? AS value
      FROM content
    ), log as (
      INSERT INTO logs (id, content_id, created_by, created_at)
      VALUES (?, ?, ?, now())
    )
    SELECT *
    FROM content
  """

  val Update = """
    WITH content AS (
      UPDATE contents
      SET version = version + 1, starred = ?,  updated_at = now()
      WHERE status = 1
        AND id = ?
      RETURNING id, version, title, description
    ), selectFeatured AS (
      UPDATE select_featured_contents
      SET position = ?, icon_type = ?, href_slug = ?
      WHERE content_id = (SELECT id FROM content)
      RETURNING href_slug
    ), translations as (
      UPDATE translations
      SET value =
        CASE
          WHEN id = (SELECT title FROM content) THEN ?
          WHEN id = (SELECT description FROM content) THEN ?
          ELSE value
        END
    ), log as (
      UPDATE logs
      SET updated_by = ?, updated_at = now()
      WHERE content_id = ?
    )
    SELECT *
    FROM content
  """

  val Delete = """
    UPDATE contents
    SET status = 0
    WHERE id = ?
      AND version = ?
  """

  val SelectById = """
    SELECT *
    FROM contents, select_featured_contents
    WHERE status = 1
      AND id = select_featured_contents.content_id
      AND content_group_id = ?
  """

  val SelectByLanguage = """
    SELECT  contents.id,
            contents.translated_id,
            contents.version,
            contents.content_type,
            contents.primary_tag,
            contents.starred,
            contents.status,
            contents.created_at,
            contents.updated_at,
            select_featured_contents.position AS select_featured_position,
            select_featured_contents.icon_type AS select_featured_icon_type,
            title.value AS title,
            description.value AS description,
            select_featured_contents.href_slug AS select_featured_href_slug
    FROM contents
    LEFT JOIN select_featured_contents on contents.id = select_featured_contents.content_id
    LEFT JOIN translations AS title on title.id = contents.title
    LEFT JOIN translations AS description on description.id = contents.description
    WHERE select_featured_contents.content_id = contents.id
    AND title.language_id = (SELECT id FROM languages WHERE name = ?)
  """

  /**
   * Find a single entry by ID.
   *
   * @param id the 128-bit UUID, as a byte array, to search for.
   * @return an optional RowData object containing the results
   */
  def find(id: Array[Byte]): Future[IndexedSeq[RowData]] = {
    pool.sendPreparedStatement(SelectById, Array(id)).map(_.rows.get)
  }

  def findByLanguage(language: String): Future[ResultSet] = {
    pool.sendPreparedStatement(SelectByLanguage, Array(language)).map { queryResult =>
      queryResult.rows.get
    }
  }

  /**
   * Inserts a new text.
   */
  def insert(id: Array[Byte],
             translated_id: Array[Byte],
             title: String,
             slug: String,
             description: String,
             href_slug: String,
             position: Integer,
             icon_type: String,
             primary_tag: String,
             starred: Integer,
             content_type: String,
             language: String,
             user_id: Array[Byte]): Future[Option[Long]] = {
    pool.sendPreparedStatement(Insert, Array(
      id, translated_id, UUID.random, slug, UUID.random, content_type, primary_tag, starred, new DateTime, new DateTime,
      href_slug, position, icon_type, language, title, language, description, UUID.random, id, user_id)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("version").asInstanceOf[Long])
            case None => None
          }
          case None => None
        }
      }
    }
  }

  /**
   * Update a SelectFeatured row.
   *
   * @return id of the saved/new audio.
   */
  def update(id: Array[Byte],
             title: String,
             slug: String,
             description: String,
             href_slug: String,
             position: Integer,
             icon_type: String,
             primary_tag: String,
             starred: Integer,
             user_id: Array[Byte]): Future[Option[Long]] = {
    pool.sendPreparedStatement(Update, Array(
      starred, id, position, icon_type, href_slug, title, description, user_id, id)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("version").asInstanceOf[Long])
            case None => None
          }
          case None => None
        }
      }
    }
  }

  /**
   * Delete a Text from the database.
   *
   * @param id the uuid of the text to delete
   * @return a boolean indicating whether the operation was successful
   */
  def delete(id: Array[Byte]): Future[Boolean] = {
    pool.sendPreparedStatement(Delete, Array(id)).map(_.rowsAffected > 0)
  }
}
