package services.datasource

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports._
import webcrank.password._
import scala.concurrent.Future
import org.joda.time.LocalDate
import play.api.Logger

import helpers.UUID

object SlideTDG extends TableDataGateway {

  val Insert = """
    WITH content AS (
      INSERT INTO contents (id, translated_id, version, title, slug, description, content_type, status, created_at, updated_at)
      VALUES (?, ?, 1, ?, ?, ?, ?, 1, ?, ?)
      RETURNING id, translated_id, version, title, description
    ), slide AS (
      INSERT INTO slide_contents (content_id, position, slide_background, content, slide_url, header_colour)
      SELECT id, ? AS position, ? AS slide_background, ? AS content, ? AS slide_url, ? AS header_colour
      FROM content
      RETURNING position, slide_background, content, slide_url
    ), translations as (
      INSERT INTO translations (id, language_id, value)
      SELECT title, (SELECT id FROM languages WHERE name = ?), ? AS value
      FROM content
      UNION ALL
      SELECT description, (SELECT id FROM languages WHERE name = ?), ? AS value
      FROM content
    ), log as (
      INSERT INTO logs (id, content_id, created_by, created_at)
      VALUES (?, ?, ?, now())
    )
    SELECT *
    FROM content
  """

  val Update = """
    WITH content AS (
      UPDATE contents
      SET version = version + 1, updated_at = now()
      WHERE status = 1
        AND id = ?
      RETURNING id, version, title, description
    ), slide AS (
      UPDATE slide_contents
      SET position = ?, slide_background = ?, content = ?, slide_url = ?, header_colour = ?
      WHERE content_id = (SELECT id FROM content)
    ), translations as (
      UPDATE translations
      SET value =
        CASE
          WHEN id = (SELECT title FROM content) THEN ?
          WHEN id = (SELECT description FROM content) THEN ?
          ELSE value
        END
    ), log as (
      UPDATE logs
      SET updated_by = ?, updated_at = now()
      WHERE content_id = ?
    )
    SELECT *
    FROM content
  """

  val SelectByLanguage = """
    SELECT  contents.id,
            contents.translated_id,
            contents.slug,
            contents.version,
            contents.content_type,
            primary_tag,
            starred,
            contents.status,
            contents.created_at,
            contents.updated_at,
            title.value AS title,
            description.value AS description,
            slide_contents.position as slide_position,
            slide_contents.slide_background,
            slide_contents.content as slide_content,
            slide_contents.slide_url,
            slide_contents.header_colour
    FROM contents
    LEFT JOIN slide_contents on contents.id = slide_contents.content_id
    LEFT JOIN translations AS title on title.id = contents.title
    LEFT JOIN translations AS description on description.id = contents.description
    WHERE slide_contents.content_id = contents.id
    AND title.language_id = (SELECT id FROM languages WHERE name = ?)
  """

  val Delete = """
    UPDATE contents
    SET status = 0
    WHERE id = ?
      AND version = ?
  """

  def findByLanguage(language: String): Future[ResultSet] = {
    pool.sendPreparedStatement(SelectByLanguage, Array(language)).map { queryResult =>
      queryResult.rows.get
    }
  }

  /**
   * Inserts a new slide.
   */
  def insert(id: Array[Byte],
             translated_id: Array[Byte],
             title: String,
             slug: String,
             description: String,
             position: Integer,
             content_type: String,
             slide_background: String,
             content: String,
             slide_url: String,
             header_colour: String,
             language: String,
             user_id: Array[Byte]): Future[Option[Long]] = {
    pool.sendPreparedStatement(Insert, Array(
      id, translated_id, UUID.random, slug, UUID.random, content_type, new DateTime, new DateTime,
      position, slide_background, content, slide_url, header_colour, language, title, language, description, UUID.random, id, user_id)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("version").asInstanceOf[Long])
            case None => None
          }
          case None => None
        }
      }
    }
  }

  /**
   * Update a Slide row.
   *
   * @return id of the saved/new slide.
   */
  def update(id: Array[Byte],
             title: String,
             slug: String,
             description: String,
             position: Integer,
             slide_background: String,
             content: String,
             slide_url: String,
             header_colour: String,
             user_id: Array[Byte]): Future[Option[Long]] = {
    pool.sendPreparedStatement(Update, Array(
      id, position, slide_background, content, slide_url, header_colour, title, description, user_id, id)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("version").asInstanceOf[Long])
            case None => None
          }
          case None => None
        }
      }
    }
  }

  /**
   * Delete a Slide from the database.
   *
   * @param id the uuid of the slide to delete
   * @return a boolean indicating whether the operation was successful
   */
  def delete(id: Array[Byte]): Future[Boolean] = {
    pool.sendPreparedStatement(Delete, Array(id)).map(_.rowsAffected > 0)
  }
}
