package services.datasource

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import scala.concurrent.Future
import play.api.Logger
import helpers.UUID

trait TableDataGateway {

  /**
   * The most important definition for any TDG object. Use the pool to send
   * database queries and prepared statements.
   *
   * @return the connection pool to be used for making database queries
   */
  def pool: Connection = DataSource.pool

  /**
   * These values are used by the below finder functions, but should be overriden
   * in the implementing class to actually, you know, do something.
   */
  val SelectAll = ""
  val SelectOne = ""

  /*
    The Finder functions!

    These functions operate the select queries, finding projects and returning
    resultsets or rowdata.
   */

  /**
   * Find all UserEntitys.
   */
  def list: Future[ResultSet] = {
    pool.sendQuery(SelectAll).map { queryResult =>
      queryResult.rows.get
    }
  }

  /**
   * Find a single entry by ID.
   *
   * @param id the 128-bit UUID, as a byte array, to search for.
   * @return an optional RowData object containing the results
   */
  def findById(id: Array[Byte]): Future[Option[RowData]] = {
    for {
      queryResult <- pool.sendPreparedStatement(SelectOne, Array(id))
      maybeRow <- Future { queryResult.rows match {
        case Some(rows) => {
          if (queryResult.rowsAffected > 0) {
            Some(rows.apply(0))
          }
          else {
            None
          }
        }
        case None => None
      }}
    } yield maybeRow
  }

}
