package services.datasource

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports._
import webcrank.password._
import scala.concurrent.Future
import org.joda.time.LocalDate
import play.api.Logger

import helpers.UUID

object TagLineTDG extends TableDataGateway {

  val Insert = """
    WITH content AS (
      INSERT INTO contents (id, translated_id, version, title, slug, description, content_type, status, created_at, updated_at)
      VALUES (?, ?, 1, ?, ?, ?, ?, 1, ?, ?)
      RETURNING id, translated_id, version, title, description
    ), tag_line AS (
      INSERT INTO tag_lines_contents (position, content_id)
      SELECT ? AS position, (SELECT id FROM content)
    ), translations as (
      INSERT INTO translations (id, language_id, value)
      SELECT title, (SELECT id FROM languages WHERE name = ?), ? AS value
      FROM content
      UNION ALL
      SELECT description, (SELECT id FROM languages WHERE name = ?), ? AS value
      FROM content
    ), log as (
      INSERT INTO logs (id, content_id, created_by, created_at)
      VALUES (?, ?, ?, now())
    )
    SELECT *
    FROM content
  """

  val Update = """
    WITH content AS (
      UPDATE contents
      SET version = version + 1, updated_at = now()
      WHERE status = 1
        AND id = ?
      RETURNING id, version, title, description
    ), tagLine AS (
      UPDATE tag_lines_contents
      SET position = ?
      WHERE content_id = (SELECT id FROM content)
    ), translations as (
      UPDATE translations
      SET value =
        CASE
          WHEN id = (SELECT title FROM content) THEN ?
          WHEN id = (SELECT description FROM content) THEN ?
          ELSE value
        END
    ), log as (
      UPDATE logs
      SET updated_by = ?, updated_at = now()
      WHERE content_id = ?
    )
    SELECT *
    FROM content
  """

  val Delete = """
    UPDATE contents
    SET status = 0
    WHERE id = ?
      AND version = ?
  """

  val SetPosition = """
    UPDATE tag_lines_contents
    SET position = ?
    WHERE content_id = ?
  """

  override val SelectAll = """
    SELECT  contents.id,
            contents.translated_id,
            contents.version,
            content_type,
            contents.status,
            contents.created_at,
            contents.updated_at,
            tag_lines_contents.position,
            title.value AS title,
            description.value AS description
    FROM contents
    LEFT JOIN tag_lines_contents on contents.id = tag_lines_contents.content_id
    LEFT JOIN translations AS title on title.id = contents.title
    LEFT JOIN translations AS description on description.id = contents.description
    WHERE status = 1
    AND tag_lines_contents.content_id = contents.id
    AND title.language_id = (SELECT id FROM languages WHERE name = 'english')
  """

  val SelectByLanguage = """
    SELECT  contents.id,
            contents.translated_id,
            contents.version,
            content_type,
            contents.status,
            contents.created_at,
            contents.updated_at,
            tag_lines_contents.position AS tag_line_position,
            title.value AS title,
            description.value AS description
    FROM contents
    LEFT JOIN tag_lines_contents on contents.id = tag_lines_contents.content_id
    LEFT JOIN translations AS title on title.id = contents.title
    LEFT JOIN translations AS description on description.id = contents.description
    WHERE tag_lines_contents.content_id = contents.id
    AND title.language_id = (SELECT id FROM languages WHERE name = ?)
  """

  override val SelectOne = """
    SELECT *
    FROM contents, tag_lines_contents
    WHERE status = 1
      AND id = ?
      AND version = ?
      AND id = content_id
  """

  val SelectById = """
    SELECT *
    FROM contents, tag_lines_contents
    WHERE status = 1
      AND id = tag_lines_contents.content_id
      AND content_group_id = ?
  """

  /**
   * Find a single entry by ID.
   *
   * @param id the 128-bit UUID, as a byte array, to search for.
   * @return an optional RowData object containing the results
   */
  def find(id: Array[Byte]): Future[IndexedSeq[RowData]] = {
    Logger.debug("[TagLineTDG.findById]")
    pool.sendPreparedStatement(SelectById, Array(id)).map(_.rows.get)
  }

  def findByLanguage(language: String): Future[ResultSet] = {
    pool.sendPreparedStatement(SelectByLanguage, Array(language)).map { queryResult =>
      queryResult.rows.get
    }
  }

  def setPosition(id: Array[Byte], position: Int): Future[Boolean] = {
    pool.sendPreparedStatement(SetPosition, Array(position, id)).map(_.rowsAffected > 0)
  }

  /**
   * Update a TagLine row.
   *
   * @return id of the saved/new audio.
   */
  def update(id: Array[Byte],
             title: String,
             slug: String,
             description: String,
             position: Integer,
             user_id: Array[Byte]): Future[Option[Long]] = {
    pool.sendPreparedStatement(Update, Array(
      id, position, title, description, user_id, id)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("version").asInstanceOf[Long])
            case None => {
              Logger.debug(queryResult.statusMessage)
              None
            }
          }
          case None => {
            Logger.debug(queryResult.statusMessage)
            None
          }
        }
      }
    }
  }

  /**
   * Inserts a new text.
   */
  def insert(id: Array[Byte],
             translated_id: Array[Byte],
             title: String,
             slug: String,
             description: String,
             position: Integer,
             content_type: String,
             language: String,
             user_id: Array[Byte]): Future[Option[Long]] = {
    Logger.debug("[TagLineTDG.save] - It's an Insert.")
    pool.sendPreparedStatement(Insert, Array(
      id, translated_id, UUID.random, slug, UUID.random, content_type, new DateTime, new DateTime,
      position, language, title, language, description, UUID.random, id, user_id)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => {
              Logger.debug("[TagLineTDG.save] - Saved TagLine, now at version " + row("version").asInstanceOf[Long].toString)
              Some(row("version").asInstanceOf[Long])
            }
            case None => {
              Logger.debug(queryResult.statusMessage)
              None
            }
          }
          case None => {
            Logger.debug(queryResult.statusMessage)
            None
          }
        }
      }
    }
  }

  /**
   * Delete a Text from the database.
   *
   * @param id the uuid of the text to delete
   * @return a boolean indicating whether the operation was successful
   */
  def delete(id: Array[Byte]): Future[Boolean] = {
    Logger.debug("[TagLineTDG.delete] - Performing Delete.")
    pool.sendPreparedStatement(Delete, Array(id)).map(_.rowsAffected > 0)
  }
}
