package services.datasource

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports._
import webcrank.password._
import scala.concurrent.Future
import org.joda.time.LocalDate
import play.api.Logger

import helpers.UUID

object TagTDG extends TableDataGateway {

  val Insert = """
    WITH tag AS (
      INSERT INTO tags (id, version, name, count, created_at, updated_at)
      VALUES (?, 1, ?, 0, ?, ?)
      RETURNING id, version, name, count
    ), translations AS (
    INSERT INTO translations (id, language_id, value)
    SELECT name, (SELECT id FROM languages WHERE name = ?), ? AS value
    FROM tag
    )
    SELECT *
    FROM tag
  """

  val Update = """
    WITH tag AS (
      UPDATE tags
      SET version = version + 1, updated_at = now()
      WHERE id = ?
      RETURNING id, version, name
    ), translations AS (
      UPDATE translations
      SET value = ?
      WHERE id = (SELECT name FROM tag)
    )
    SELECT version
    FROM tag
  """

  val Delete = """
    WITH tag AS (
      DELETE FROM tags
      WHERE id = ?
    ), deleted AS (
      DELETE FROM tags_contents
      WHERE tag_id = ?
      RETURNING tag_id
    )
    SELECT count(tag_id) FROM deleted
  """

  override val SelectAll = """
    SELECT  tags.id,
            tags.version,
            tags.count,
            tags.created_at,
            tags.updated_at,
            name.value AS name
    FROM tags
    LEFT JOIN translations AS name on name.id = tags.name
  """

  override val SelectOne = """
    SELECT  tags.id,
            tags.version,
            tags.count,
            tags.created_at,
            tags.updated_at,
            name.value AS name,
    FROM tags
    LEFT JOIN translations AS name on name.id = tags.name
    WHERE id = ?
      AND version = ?
  """

  val SelectById = """
    SELECT  tags.id,
            tags.version,
            tags.count,
            tags.created_at,
            tags.updated_at,
            name.value AS name
    FROM tags
    LEFT JOIN translations AS name on name.id = tags.name
    WHERE tags.id = ?
    LIMIT 1
  """

  val SelectByName = """
    SELECT  tags.id,
            tags.version,
            tags.count,
            tags.created_at,
            tags.updated_at,
            name.value AS name
    FROM tags
    LEFT JOIN translations AS name on name.id = tags.name
    WHERE LOWER(name.value) LIKE LOWER(?)
      AND language_id = (SELECT id FROM languages WHERE name = ?)
    LIMIT 1
  """

  val SelectByLanguage = """
    SELECT  tags.id,
            tags.version,
            tags.count,
            tags.created_at,
            tags.updated_at,
            name.value AS name
    FROM tags
    LEFT JOIN translations AS name on name.id = tags.name
    WHERE language_id = (SELECT id FROM languages WHERE name = ?)
  """

  val SelectByTagged = """
    WITH tag_ids AS (
      SELECT tag_id
      FROM tags_contents
      WHERE tags_contents.content_id = ?
    )
    SELECT  tags.id,
            tags.version,
            tags.count,
            tags.created_at,
            tags.updated_at,
            name.value AS name
    FROM tags
    LEFT JOIN translations AS name on name.id = tags.name
    LEFT JOIN tag_ids on tag_ids.tag_id = tags.id
    WHERE tags.id = tag_ids.tag_id
  """

  def SelectByManyTagged(numIds: Int): String = {
    var whereClause = ""

    for(i <- 0 to (numIds - 1)) {
      whereClause += "?"
      if (i != (numIds - 1)) whereClause += ", "
    }

    """
    WITH tag_ids AS (
      SELECT tag_id
      FROM tags_contents
      WHERE tags_contents.content_id IN (""" + whereClause + """)
    )
    SELECT  tags.id,
            tags.version,
            tags.count,
            tags.created_at,
            tags.updated_at,
            name.value AS name
    FROM tags
    LEFT JOIN translations AS name on name.id = tags.name
    LEFT JOIN tag_ids on tag_ids.tag_id = tags.id
    WHERE tags.id = tag_ids.tag_id
    """
  }

  def SelectByTags(tags: Array[String]): String = {
    var whereClause = ""

    for(i <- 0 to (tags.length - 1)) {
      whereClause += "'" + tags(i) + "'"
      if (i != (tags.length - 1)) whereClause += ", "
    }

    """
    WITH id_count as (
      SELECT  contents.id,
              COUNT(contents.id) AS count
      FROM contents
      INNER JOIN tags_contents on tags_contents.content_id = contents.id
      INNER JOIN tags on tags_contents.tag_id = tags.id
      LEFT JOIN text_contents on contents.id = text_contents.content_id
      LEFT JOIN video_contents on contents.id = video_contents.content_id
      LEFT JOIN link_contents on contents.id = link_contents.content_id
      LEFT JOIN audio_contents on contents.id = audio_contents.content_id
      LEFT JOIN translations AS name on name.id = tags.name
      LEFT JOIN translations AS title on title.id = contents.title
      WHERE name.value IN (""" + whereClause + """)
      AND (text_contents.content_id = contents.id or video_contents.content_id = contents.id or link_contents.content_id = contents.id or audio_contents.content_id = contents.id)
      AND title.language_id = (SELECT id FROM languages WHERE name = ?)
      GROUP BY contents.id
    ), content_ids AS (
      SELECT  DISTINCT ON (contents.id)
              contents.id,
              contents.translated_id
      FROM contents
      INNER JOIN tags_contents on tags_contents.content_id = contents.id
      INNER JOIN tags on tags_contents.tag_id = tags.id
      LEFT JOIN text_contents on contents.id = text_contents.content_id
      LEFT JOIN video_contents on contents.id = video_contents.content_id
      LEFT JOIN link_contents on contents.id = link_contents.content_id
      LEFT JOIN audio_contents on contents.id = audio_contents.content_id
      WHERE contents.id IN (SELECT id FROM id_count WHERE count = '""" + tags.length + """')
    ), tag_ids AS (
      SELECT tag_id
      FROM tags_contents
      WHERE tags_contents.content_id IN (SELECT id FROM content_ids)
    )
    SELECT  tags.id,
            tags.version,
            tags.count,
            tags.created_at,
            tags.updated_at,
            name.value AS name
    FROM tags
    LEFT JOIN translations AS name on name.id = tags.name
    LEFT JOIN tag_ids on tag_ids.tag_id = tags.id
    WHERE tags.id = tag_ids.tag_id
    """
  }

  val TagContent = """
    WITH tag_content AS (
      INSERT INTO tags_contents (content_id, tag_id, created_at)
      SELECT ? AS content_id, ? AS tag_id, ? AS created_at
      WHERE NOT EXISTS (
        SELECT content_id, tag_id, created_at
        FROM tags_contents
        WHERE content_id = ?
          AND tag_id = ?
      )
      RETURNING tag_id
    )
    UPDATE tags
    SET count = count + 1
    WHERE tags.id = (SELECT tag_content.tag_id FROM tag_content)
  """

  val UntagContent = """
    WITH tag_content AS (
      DELETE FROM
      tags_contents
      WHERE content_id = ?
      RETURNING tag_id
    )
    UPDATE tags
    SET count = count - 1
    WHERE tags.id IN (SELECT tag_content.tag_id FROM tag_content)
  """

  /**
   * Find a single entry by ID.
   *
   * @param id the 128-bit UUID, as a byte array, to search for.
   * @return an optional RowData object containing the results
   */
  def find(id: Array[Byte]): Future[IndexedSeq[RowData]] = {
    pool.sendPreparedStatement(SelectById, Array(id)).map(_.rows.get)
  }

  def findByName(name: String, language: String): Future[IndexedSeq[RowData]] = {
    pool.sendPreparedStatement(SelectByName, Array(name, language)).map(_.rows.get)
  }

  def findByLanguage(language: String): Future[ResultSet] = {
    pool.sendPreparedStatement(SelectByLanguage, Array(language)).map { queryResult =>
      queryResult.rows.get
    }
  }

  def findByTagged(content_id: Array[Byte]): Future[ResultSet] = {
    pool.sendPreparedStatement(SelectByTagged, Array(content_id)).map { queryResult =>
      queryResult.rows.get
    }
  }

  def findByManyTagged(content_ids: Array[Array[Byte]]): Future[ResultSet] = {
    var numIds = content_ids.length

    pool.sendPreparedStatement(SelectByManyTagged(numIds), content_ids).map { queryResult =>
      queryResult.rows.get
    }
  }

  def findByTags(tags: Array[String], language: String): Future[ResultSet] = {
    pool.sendPreparedStatement(SelectByTags(tags.map(tag => tag.replace("'", "''"))), Array(language)).map { queryResult =>
      queryResult.rows.get
    }
  }

  /**
   * Update a Text row.
   *
   * @return id of the saved/new text.
   */
  def update(id: Array[Byte],
             name: String): Future[Option[Long]] = {
    pool.sendPreparedStatement(Update, Array(
      id, name)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("version").asInstanceOf[Long])
            case None => {
              Logger.debug(queryResult.statusMessage)
              None
            }
          }
          case None => {
            Logger.debug(queryResult.statusMessage)
            None
          }
        }
      }
    }
  }

  /**
   * Inserts a new tag.
   */
  def insert(id: Array[Byte],
             name: String,
             language: String): Future[Option[Long]] = {
    pool.sendPreparedStatement(Insert, Array(
      id, UUID.random, new DateTime, new DateTime, language, name)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("version").asInstanceOf[Long])
            case None => {
              Logger.debug(queryResult.statusMessage)
              None
            }
          }
          case None => {
            Logger.debug(queryResult.statusMessage)
            None
          }
        }
      }
    }
  }

  def tagContent(tag_id: Array[Byte], content_id: Array[Byte]): Future[Boolean] = {
    pool.sendPreparedStatement(TagContent, Array(content_id, tag_id, new DateTime, content_id, tag_id)).map(_.rowsAffected > 0)
  }

  def untagContent(content_id: Array[Byte]): Future[Boolean] = {
    pool.sendPreparedStatement(UntagContent, Array(content_id)).map(_.rowsAffected > 0)
  }

  /**
   * Delete a tag from the database.
   *
   * @param id the uuid of the tag to delete
   * @return a boolean indicating whether the operation was successful
   */
  def delete(id: Array[Byte]): Future[Boolean] = {
    for {
      queryResult <- pool.sendPreparedStatement(Delete, Array(id, id))
    } yield { queryResult.rowsAffected > 0 }
  }
}
