package services.datasource

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports._
import webcrank.password._
import scala.concurrent.Future
import org.joda.time.LocalDate
import play.api.Logger
import helpers.UUID

object UserTDG extends TableDataGateway {

  /*
    The SQL queries!

    Write all SQL queries that need for interacting with the users table here.
   */

  val Insert = """
    INSERT INTO users (id, version, email, password, status, created_at, updated_at)
    VALUES (?, ?, ?, ?, 1, ?, ?)
    RETURNING version
  """

  val Update = """
    UPDATE users
    SET version = ?, email = ?, password = ?, updated_at = ?
    WHERE id = ?
      AND status = 1
    RETURNING version
  """

  override val SelectAll = """
    SELECT id, version, email, password, status, created_at, updated_at
    FROM users
    WHERE status = 1
  """

  val SelectOneUser = """
    SELECT id, version, email, password, status, created_at, updated_at
    FROM users
    WHERE email = ?
      AND password = ?
      AND status = 1
  """

  val SelectByEmail = """
    SELECT id, version, email, password, status, created_at, updated_at
    FROM users
    WHERE email = ?
      AND status = 1
  """

  override val SelectOne = """
    SELECT id, version, email, password, status, created_at, updated_at
    FROM users
    WHERE id = ?
      AND status = 1
  """

  val Delete = """
    DELETE FROM users
    WHERE id = ?
  """

  def authenticate(email: String, password: String): Future[Option[RowData]] = {
    for {
      queryResult <- pool.sendPreparedStatement(SelectOneUser, Array(email, password))
      maybeRow <- Future { queryResult.rows match {
        case Some(rows) => {
          rows.headOption match {
            case Some(rowData) => Some(rowData)
            case None => None
          }
        }
        case None => None
      }}
    } yield maybeRow
  }

  def find(id: Array[Byte]): Future[IndexedSeq[RowData]] = {
    pool.sendPreparedStatement(SelectOne, Array(id)).map(_.rows.get)
  }

  /**
   * Find a single UserEntity by e-mail address.
   *
   * @param email the email address to search by
   * @return the future optional RowData returned from the database
   */
  def findByEmail(email: String): Future[IndexedSeq[RowData]] = {
    pool.sendPreparedStatement(SelectByEmail, Array(email)).map(_.rows.get)
  }

  /**
   * Save a User row.
   *
   * @return id of the saved/new user.
   */
  def update(id: Array[Byte],
             version: Long,
             email: String,
             password: String): Future[Option[Long]] = {
    pool.sendPreparedStatement(Update, Array(version, email, password, new DateTime, id)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("version").asInstanceOf[Long])
            case None => {
              Logger.debug(queryResult.statusMessage)
              None
            }
          }
          case None => {
            Logger.debug(queryResult.statusMessage)
            None
          }
        }
      }
    }
  }

  /**
   * Inserts a new user.
   */
  def insert(id: Array[Byte],
             version: Long,
             email: String,
             password: String): Future[Option[Long]] = {
    pool.sendPreparedStatement(Insert, Array(id, version, email, password, new DateTime, new DateTime)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("version").asInstanceOf[Long])
            case None => {
              Logger.debug(queryResult.statusMessage)
              None
            }
          }
          case None => {
            Logger.debug(queryResult.statusMessage)
            None
          }
        }
      }
    }
  }

  /**
   *
   * @param id
   * @return
   */
  def delete(id: String, version: Long): Future[Boolean] = {
    for {
      queryResult <- pool.sendPreparedStatement(Delete, Array(id, version))
    } yield { queryResult.rowsAffected > 0 }
  }
}
