package services.datasource

import com.github.mauricio.async.db.{RowData, Connection, ResultSet}
import com.github.mauricio.async.db.util.ExecutorServiceUtils.CachedExecutionContext
import com.github.nscala_time.time.Imports._
import webcrank.password._
import scala.concurrent.Future
import org.joda.time.LocalDate
import play.api.Logger

import helpers.UUID

object VideoTDG extends TableDataGateway {

  val Insert = """
    WITH content AS (
      INSERT INTO contents (id, translated_id, version, title, slug, description, content_type, primary_tag, starred, status, created_at, updated_at)
      VALUES (?, ?, 1, ?, ?, ?, ?, ?, ?, 1, ?, ?)
      RETURNING id, translated_id, version, title, description
    ), video AS (
      INSERT INTO video_contents (content_id, video_type, video_url, long_description)
      SELECT content.id, ? AS video_type, ? AS video_url, ? AS long_description
      FROM content
      RETURNING video_url, long_description
    ), translations as (
      INSERT INTO translations (id, language_id, value)
      SELECT title, (SELECT id FROM languages WHERE name = ?), ? AS value
      FROM content
      UNION ALL
      SELECT description, (SELECT id FROM languages WHERE name = ?), ? AS value
      FROM content
      UNION ALL
      SELECT long_description, (SELECT id FROM languages WHERE name = ?), ? AS value
      FROM video
    ), log as (
      INSERT INTO logs (id, content_id, created_by, created_at)
      VALUES (?, ?, ?, now())
    )
    SELECT *
    FROM content
  """

  val Update = """
    WITH content AS (
      UPDATE contents
      SET version = version + 1, primary_tag = ?, starred = ?, slug = ?, updated_at = now()
      WHERE status = 1
        AND id = ?
      RETURNING id, version, title, description
    ), video as (
      UPDATE video_contents
      SET video_type = ?, video_url = ?
      WHERE content_id = (SELECT content.id FROM content)
    ), long_description AS (
      SELECT long_description
      FROM video_contents
      WHERE content_id = ?
    ), translations as (
      UPDATE translations
      SET value =
        CASE
          WHEN id = (SELECT title FROM content) THEN ?
          WHEN id = (SELECT description FROM content) THEN ?
          WHEN id = (SELECT long_description FROM long_description) THEN ?
          ELSE value
        END
    ), log as (
      UPDATE logs
      SET updated_by = ?, updated_at = now()
      WHERE content_id = ?
    )
    SELECT *
    FROM content
  """

  val Delete = """
    UPDATE contents
    SET status = 0
    WHERE id = ?
      AND version = ?
  """

  /**
   * Update a Video row.
   *
   * @return id of the saved/new video.
   */
  def update(id: Array[Byte],
             title: String,
             slug: String,
             description: String,
             long_description: String,
             video_type: String,
             video_url: String,
             primary_tag: String,
             starred: Integer,
             user_id: Array[Byte]): Future[Option[Long]] = {
    pool.sendPreparedStatement(Update, Array(
      primary_tag, starred, slug, id, video_type, video_url, id, title, description, long_description, user_id, id)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("version").asInstanceOf[Long])
            case None => {
              Logger.debug(queryResult.statusMessage)
              None
            }
          }
          case None => {
            Logger.debug(queryResult.statusMessage)
            None
          }
        }
      }
    }
  }

  /**
   * Inserts a new video.
   */
  def insert(id: Array[Byte],
             translated_id: Array[Byte],
             title: String,
             slug: String,
             description: String,
             long_description: String,
             content_type: String,
             primary_tag: String,
             starred: Integer,
             video_type: String,
             video_url: String,
             language: String,
             user_id: Array[Byte]): Future[Option[Long]] = {
    pool.sendPreparedStatement(Insert, Array(
      id, translated_id, UUID.random, slug, UUID.random, content_type, primary_tag, starred, new DateTime, new DateTime,
      video_type, video_url, UUID.random, language, title, language, description, language, long_description,
      UUID.random, id, user_id)).map {
      queryResult => {
        queryResult.rows.headOption match {
          case Some(resultSet) => resultSet.headOption match {
            case Some(row) => Some(row("version").asInstanceOf[Long])
            case None => {
              Logger.debug(queryResult.statusMessage)
              None
            }
          }
          case None => {
            Logger.debug(queryResult.statusMessage)
            None
          }
        }
      }
    }
  }

  /**
   * Delete a Video from the database.
   *
   * @param id the uuid of the video to delete
   * @return a boolean indicating whether the operation was successful
   */
  def delete(id: Array[Byte]): Future[Boolean] = {
    pool.sendPreparedStatement(Delete, Array(id)).map(_.rowsAffected > 0)
  }
}
