package controllers

import play.api._
import play.api.mvc._
import play.api.Play.current
import play.api.i18n.{Messages, Lang}
import play.api.libs.concurrent.Execution.Implicits._

import scala.concurrent.Future

import models._

import helpers.{UUID, JsMessages}

object Application extends Controller with controllers.tagger.Secured {

  val messages = JsMessages.default

  val allJsMessages = Action {
    Ok(messages.all(Some("window.Messages")))
  }

  def index = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    var language = Play.current.configuration.getString("tagger.language").get
    var title = "FLS"

    val fMenuItems = MenuItem.findAllParents(language).flatMap { parentMenus =>
      Future.sequence(parentMenus.map { parentMenu =>
        for {
          cSubMenus <- MenuItem.findByParentId(parentMenu.id.string, language)
        } yield {
          (parentMenu, cSubMenus)
        }
      })
    }

    for {
      cTagLines <- TagLine.findByLanguage(language)
      cHeaderItems <- HeaderItem.findByLanguage(language)
      cSlides <- Slide.findByLanguage(language)
      cFeatured <- Featured.findByLanguage(language)
      cSelectFeatured <- SelectFeatured.findByLanguage(language)
      cHomeBlock <- HomeBlock.findByLanguage(language)
      cMenuItems <- fMenuItems
    } yield {
      Ok(views.html.index(title, cHeaderItems, cMenuItems, cTagLines, cSlides, cFeatured, cSelectFeatured, cHomeBlock.get))
    }
  }

  def search(tag_line_id: String) = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    var language = Play.current.configuration.getString("tagger.language").get
    var title = "FLS Search"

    var mUUID = if (tag_line_id != "none") { tag_line_id } else { UUID.random.string }

    val fMainTagLine: Future[Option[(TagLine, String)]] = TagLine.findById(UUID(mUUID), language).flatMap {
      case Some(content) => {
        Tag.findByTagged(content.id.bytes, language).map { cTags =>
          var tags = Seq[String]()

          for(tag <- cTags) { tags = (tags :+ tag.name) }
          Some(content.asInstanceOf[TagLine], tags.mkString(","))
        }
      }
      case None => {Future(None)}
    }

    val fMenuItems = MenuItem.findAllParents(language).flatMap { parentMenus =>
      Future.sequence(parentMenus.map { parentMenu =>
        for {
          cSubMenus <- MenuItem.findByParentId(parentMenu.id.string, language)
        } yield {
          (parentMenu, cSubMenus)
        }
      })
    }

    for {
      cHeaderItems <- HeaderItem.findByLanguage(language)
      cTagLines <- TagLine.findByLanguage(language)
      cMainTagLine <- fMainTagLine
      cHomeBlock <- HomeBlock.findByLanguage(language)
      cMenuItems <- fMenuItems
    } yield {
      Ok(views.html.search(title, cTagLines, cHeaderItems, cMainTagLine, cHomeBlock.get, cMenuItems))
    }
  }

  def content(primaryTag: String, slug: String) = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    var language = Play.current.configuration.getString("tagger.language").get
    var title = "FLS"

    val subDir = {
      if (primaryTag != "none") {
        primaryTag + "/"
      }
      else {
        ""
      }
    }

    val fMenuItems = MenuItem.findAllParents(language).flatMap { parentMenus =>
      Future.sequence(parentMenus.map { parentMenu =>
        for {
          cSubMenus <- MenuItem.findByParentId(parentMenu.id.string, language)
        } yield {
          (parentMenu, cSubMenus)
        }
      })
    }

    for {
      cTagLines <- TagLine.findByLanguage(language)
      cHeaderItems <- HeaderItem.findByLanguage(language)
      cContent <- ContentModel.findBySlug(subDir + slug)
      tags <- Tag.findByTagged(cContent.get.id.bytes, language)
      cHomeBlock <- HomeBlock.findByLanguage(language)
      cMenuItems <- fMenuItems
    } yield {
      Ok(views.html.content(cContent.get, tags, cTagLines, cHeaderItems, cHomeBlock.get, cMenuItems))
    }
  }

  def menuContent(id: String) = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    var language = Play.current.configuration.getString("tagger.language").get
    var title = "FLS"


    val fMenuItems = MenuItem.findAllParents(language).flatMap { parentMenus =>
      Future.sequence(parentMenus.map { parentMenu =>
        for {
          cSubMenus <- MenuItem.findByParentId(parentMenu.id.string, language)
        } yield {
          (parentMenu, cSubMenus)
        }
      })
    }

    for {
      cTagLines <- TagLine.findByLanguage(language)
      cHeaderItems <- HeaderItem.findByLanguage(language)
      cContent <- ContentModel.find(UUID(id), language)
      tags <- Tag.findByTagged(cContent.get.id.bytes, language)
      cHomeBlock <- HomeBlock.findByLanguage(language)
      cMenuItems <- fMenuItems
    } yield {
      if (cContent.get.contentType == "link") {
        Redirect(cContent.get.asInstanceOf[Link].linkUrl)
      } else {
        Ok(views.html.content(cContent.get, tags, cTagLines, cHeaderItems, cHomeBlock.get, cMenuItems))
      }
    }
  }

  /**
   * Change language of site.
   */
  def language(lang: String) = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    val referrer = request.headers.get(REFERER).getOrElse("/")
    Future(Redirect(referrer).withLang(Lang(lang)).asInstanceOf[SimpleResult])
  }

  /**
   * Sends file to client from the server's "uploads" directory.
   */
  def sendImage(file: String) = MaybeAuthenticatedAsync { implicit maybeUser => implicit request =>
    Future {
      Ok.sendFile(new java.io.File(Play.current.configuration.getString("uploads.outFile").get + file))
    }
  }

}
